#ifndef NOMORE_OVERLAY_LIST_H_INCLUDED
#define NOMORE_OVERLAY_LIST_H_INCLUDED

#include <algorithm> // std::copy, std::swap
#include <cstddef>  // std::size_t
#include <node.h>
#include <cassert>

#if defined (_MSC_VER) && _MSC_VER <= 1300
namespace std {
  using ::size_t;
}
#endif


namespace NS_NOMORE {

class StaticOverlayList {
public:
  StaticOverlayList() : index_(0), indexSize_(0) {}
  explicit StaticOverlayList(std::size_t indexSize)
    : index_(new char[indexSize])
    , indexSize_(indexSize) {
  }
  ~StaticOverlayList() {
    delete [] index_;
  }

  // only grow
  void resize(std::size_t newIndexSize) {
    if (indexSize_ < newIndexSize) {
      char* temp = new char[newIndexSize];
      std::copy(index_, index_ + indexSize_, temp);
      std::fill(temp + indexSize_, temp + newIndexSize, 0);
      indexSize_ = newIndexSize;
      std::swap(temp, index_);
      delete [] temp;
    }
  }

  // node n is overlayed by Color::ColorValue c
  void add(const Node& n, Color::ColorValue c) {
    assert(n.getId() < static_cast<long>(indexSize_));
    index_[n.getId()] |= c;
  }

  // is n overlayed by c?
  bool isOverlayed(const Node& n, Color::ColorValue c) const {
    assert(n.getId() < static_cast<long>(indexSize_));
    return (index_[n.getId()] & c) != 0;
  }

  // returns true if n is overlayed by some Color
  bool hasOverlay(const Node& n) const {
    assert(n.getId() < static_cast<long>(indexSize_));
    return index_[n.getId()] != 0;
  }

  // clears this overlay list
  void clear() {
    std::fill(index_, index_ + indexSize_, 0);
  }
private:
  StaticOverlayList(const StaticOverlayList&);
  StaticOverlayList& operator=(const StaticOverlayList&);
  char* index_;
  std::size_t indexSize_;
};



}

#endif
