/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <operators/unfounded_loops.h>
#include <graph.h>

#ifdef KDEVELOP
  #include "algorithms.h"
#else
  #include "../detail/algorithms.h"
#endif

#if defined (_MSC_VER) && _MSC_VER <= 1300
#define for if(0); else for
#endif

namespace NS_NOMORE {

///////////////////////////////////////////////////////////////////////////////
template <>
const char* getName(UnfoundedLoopsOperator*) {
  return "L";
}

template <>
Operator* createDetOp(Graph& g, UnfoundedLoopsOperator*) {
  return new UnfoundedLoopsOperator(g);
}
///////////////////////////////////////////////////////////////////////////////


  
UnfoundedLoopsOperator::UnfoundedLoopsOperator(Graph& grp) 
  : PropOperator(grp)
  , disabled_(false) {
  event::ChannelManager &em = grp.getEventManager();

  em.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);
  em.getChannelFor(event::Event<BodyNodeColoredChoicePoint>()).connect(*this);
  em.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).connect(*this);
  
  grp.checkTightness();
  em.getChannelFor(event::Event<GraphConstructionFinished>()).connect(*this);
}

///////////////////////////////////////////////////////////////////////////////
// common interface
///////////////////////////////////////////////////////////////////////////////
const char* UnfoundedLoopsOperator::getName() const {
  return NS_NOMORE::getName<UnfoundedLoopsOperator>(0);
}

void UnfoundedLoopsOperator::execute() {
  if (!disabled_) {
    bool loopFound = false;
    for (HeadNode* head = 0; !todo_.empty() && !loopFound; ) {
      head = todo_.front();
      todo_.pop_front();
      if (Color::isUncolored(head->getColor())) {
        loopFound = checkLoop(head);
      }
    }
  }
}


void UnfoundedLoopsOperator::reset() {
  todo_.clear();
  clearWFList();
}


bool UnfoundedLoopsOperator::isDisabled() const {
  return disabled_;
}

///////////////////////////////////////////////////////////////////////////////
// event handlers
///////////////////////////////////////////////////////////////////////////////
void UnfoundedLoopsOperator::handle(const GraphConstructionFinished& e) {
  assert(e.theGraph_ == &getGraph());

  if (e.theGraph_->tightness() == Graph::tight) {
    event::ChannelManager &em = getGraph().getEventManager();
    em.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
    em.getChannelFor(event::Event<BodyNodeColoredChoicePoint>()).disconnect(*this);
    em.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).disconnect(*this);
    em.getChannelFor(event::Event<GraphConstructionFinished>()).disconnect(*this);
    disabled_ = true;
  }
}

///////////////////////////////////////////////////////////////////////////////
// event handlers
///////////////////////////////////////////////////////////////////////////////
void UnfoundedLoopsOperator::handle(const BodyNodeColored& e) {
  if(e.node_->getColor() == Color::minus) {
    updateTodoList(e.node_);
    updateWFList(e.node_);
  }
    
}

void UnfoundedLoopsOperator::handle(const BodyNodeColoredChoicePoint& e) {
  if(e.node_->getColor() == Color::minus) {
    updateTodoList(e.node_);
  }
  clearWFList();
}

void UnfoundedLoopsOperator::handle(const HeadNodeColoredChoicePoint&) {
  clearWFList();
}

///////////////////////////////////////////////////////////////////////////////
// private helpers
///////////////////////////////////////////////////////////////////////////////
bool UnfoundedLoopsOperator::checkLoop(HeadNode* head) {
  Stack dfsStack;
  Stack loop;
  dfsStack.push_back(head);
  int masterComponent = head->getComponentNumber();
  while (!dfsStack.empty()) {
    head = dfsStack.back();
    dfsStack.pop_back();
    if (BodyNode* n = getWellfounded(head, masterComponent)) {
      addToWFList(head, n);
      dfsStack.insert(dfsStack.end(), loop.begin(), loop.end());
      loop.clear();
    }
    else {
      HeadNode::BodyNodeIterator aBody = head->predecessorsBegin();
      while ((aBody = pickBody(head, aBody, dfsStack, loop)) != head->predecessorsEnd() ) {
        if (HeadNode* newHead = pickHead(*aBody, masterComponent)) {
          dfsStack.push_back(head);
          dfsStack.push_back(newHead);
          break;
        }
      }
      if (aBody == head->predecessorsEnd()) { // dfs is complete
        loop.push_back(head);
        if (loopIsUnfounded(loop)) {
          color(loop);
          DETAIL::copy_if(dfsStack.begin(), dfsStack.end(), std::front_inserter(todo_), HasColor(Color::none | Color::weak_plus));
          return true;
        }
      }
    }
  }
  return false;
}


BodyNode* UnfoundedLoopsOperator::getWellfounded(HeadNode* head, int component) const {
  HeadNode::BodyNodeIterator bodyEnd = head->predecessorsEnd();
  for (HeadNode::BodyNodeIterator body = head->predecessorsBegin(); body != bodyEnd; ++body) {
    if ( (*body)->getColor() != Color::minus ) {
      BodyNode::HeadNodeIterator atomEnd = (*body)->zeroPredecessorsEnd();
      int count = 0;
      int skip = 0;
      for (BodyNode::HeadNodeIterator atom = (*body)->zeroPredecessorsBegin(); atom != atomEnd; ++atom, ++count) {
        if ( (*atom)->getColor() == Color::plus || isInWFList(*atom) ) {
            ++skip;
            continue;
        }
        if ((*atom)->getComponentNumber() != component) {
          return *body;
        }
        
      }
      if (skip == count) 
        return *body;
    }
  }
  return 0;
}

HeadNode::BodyNodeIterator UnfoundedLoopsOperator::pickBody(HeadNode* head, HeadNode::BodyNodeIterator start, const Stack& dfsStack, const Stack& loop) const {
  HeadNode::BodyNodeIterator end = head->predecessorsEnd();
  HeadNode::BodyNodeIterator foundBody = end;
  for (HeadNode::BodyNodeIterator it = start; it != end && foundBody == end; ++it) {
    if ((*it)->getColor() != Color::minus) {
      foundBody = it;
      BodyNode::HeadNodeIterator atomEnd = (*it)->zeroPredecessorsEnd();
      for (BodyNode::HeadNodeIterator atom = (*it)->zeroPredecessorsBegin(); atom != atomEnd; ++atom) {
        if (std::find_if(dfsStack.begin(), dfsStack.end(), EqualId(*atom)) != dfsStack.end()
          || std::find_if(loop.begin(), loop.end(), EqualId(*atom)) != loop.end()) {
            foundBody = end;
            break;
        }
          
      }
    }
  }
  return foundBody;
}

HeadNode* UnfoundedLoopsOperator::pickHead(BodyNode* body, int component) const {
  for (BodyNode::HeadNodeIterator it = body->zeroPredecessorsBegin(); it != body->zeroPredecessorsEnd(); ++it) {
    if ((*it)->getColor() != Color::plus && (*it)->getComponentNumber() == component
      && isInWFList(*it) == false)
      return *it;
  }
  return 0;
}

bool UnfoundedLoopsOperator::loopIsUnfounded(const Stack& loop) const {
  for (size_t i = 0; i != loop.size(); ++i) {
    if ( pickBody(loop[i], loop[i]->predecessorsBegin(), Stack(), loop) != loop[i]->predecessorsEnd()) {
      return false;
    }
  }
  return true;
}

void UnfoundedLoopsOperator::color(const Stack& loop) {
  Graph& g = getGraph();
  for (size_t i = 0; i != loop.size(); ++i) {
    if (!g.color(*loop[i], Color::minus))
      return;
  }
}

bool UnfoundedLoopsOperator::isInWFList(HeadNode* head) const {
  return wellFoundedHeads_.find(head) != wellFoundedHeads_.end();
}

void UnfoundedLoopsOperator::addToWFList(HeadNode* head, BodyNode* body) {
  wellFoundedHeads_.insert(head);
  wellFoundedBodies_.insert(WellFoundedBodyList::value_type(body, head));
}



void UnfoundedLoopsOperator::updateTodoList(BodyNode* n) {
  DETAIL::copy_if(n->successorsBegin(), n->successorsEnd(), 
    std::back_inserter(todo_), HasColor(Color::none | Color::weak_plus));
}

void UnfoundedLoopsOperator::updateWFList(BodyNode* b) {
  typedef WellFoundedBodyList::iterator Iterator;
  for (Iterator it = wellFoundedBodies_.find(b); it != wellFoundedBodies_.end(); it = wellFoundedBodies_.find(b)) {
    HeadNode* h = it->second;
    wellFoundedHeads_.erase(h);
    wellFoundedBodies_.erase(it);
    HeadNode::BodyNodeIterator bodyEnd = h->zeroSuccessorsEnd();
    for (HeadNode::BodyNodeIterator body = h->zeroSuccessorsBegin(); body != bodyEnd; ++body) {
      updateWFList(*body);
    }
  }
}

void UnfoundedLoopsOperator::clearWFList() {
  wellFoundedHeads_.clear();
  wellFoundedBodies_.clear();
}

}
