/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_RANDOM_HEURISTIC_H
#define LIBNOMORE_RANDOM_HEURISTIC_H

#include <static_heuristic.h>
#include <dynamic_heuristic.h>

#include <functional>

namespace NS_NOMORE {

//! Creating a Choice with random colors for the random heuristics.
/*!
 *  The preferred and alternative are set randomly (but not to the same 
 *  color). */
struct RandomColor {
  //! Creats the Choice with the random color.
  /*! 
   *  \param n The node for which the Choice with the random colors are 
   *           created.
   *  \return The Choice with the node n and random colors. */
  Choice createChoice(Node* n) const;
};

//! Heuristic strategy that emulates random sorted rules and atoms.
/*! 
 *  The heuristic values of all uncolored nodes set to random number between 
 *  0 and 1023. The preferred and alternative color is set randomly to 
 *  Node::preferredChoicePointColor() and Node::alternativeChoicePointColor() 
 *  or vice versa.
 *
 *  \note Can be used with head and body nodes.
 *  \note This strategy may be used with either a static  or a dynamic 
 *        heuristic. */
class RandomStrat {
public:
  //! Value type of the stategie.
  typedef long ValueType; 

  //! The greater comparator for the value type.
  typedef std::greater<ValueType> ComparatorType;

  //! Sets the heuristic value for a node.
  struct Evaluate {
    //! Evaluates the heuristic valie of the node using a tightness factor.
    /*! 
     *  \param n The node for the value computation.
     *  \param tf The tightness factor used for the heuristic value. */
    ValueType operator()(Node* n, double tf) const;
  };  

  //! Evaluator type of the strategie.
  typedef Evaluate EvaluatorType;

  //! The choice creator type used to create a Choice.
  typedef RandomColor ChoiceCreatorType;
};

//! Heuristic strategy that emulates random sorted atoms.
/*! 
 *  The heuristic values of all uncolored head nodes set to random number 
 *  between 0 and 1023. The preferred and alternative color is set randomly 
 *  to Node::preferredChoicePointColor() and 
 *  Node::alternativeChoicePointColor() or vice versa.
 *
 *  \note Can be used only with head nodes (representing atoms).
 *  \note This strategy may be used with either a static or a dynamic 
 *        heuristic. */
class RandomHeadNodeStrat {
public:
  //! Value type of the stategie.
  typedef long ValueType; 

  //! The greater comparator.
  typedef std::greater<ValueType> ComparatorType;

  //! Sets the heuristic value for a node using the tightness factor.
  struct Evaluate {
    //! Evaluates the heuristic valie of the node using a tightness factor.
    /*! 
     *  \param n The node for the value computation.
     *  \param tf The tightness factor used for the heuristic value. */
    ValueType operator()(Node* n, double tf) const;
  };  

  //! Evaluator type of the strategie.
  typedef Evaluate EvaluatorType;
  
  //! The choice creator type used to create a Choice.
  typedef RandomColor ChoiceCreatorType;
};

//! A static heuristic that uses RandomStrat as its strategy.
typedef StaticHeuristic<RandomStrat> StaticRandom;

//! A dynamic heuristic that uses RandomStrat as its strategy.
typedef DynamicHeuristic<RandomStrat> DynamicRandom;

//! A static heuristic that uses RandomHeadNodeStrat as its strategy.
typedef StaticHeuristic<RandomHeadNodeStrat> StaticRandomHeads;

//! A dynamic heuristic that uses RandomHeadNodeStrat as its strategy.
typedef DynamicHeuristic<RandomHeadNodeStrat> DynamicRandomHeads;

}
#endif
