/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_BODY_LOOKAHEAD_OPERATOR_H
#define NOMORE_BODY_LOOKAHEAD_OPERATOR_H

#include <operators/lookahead.h>
#include <heuristic.h>
#include <graph.h>
#include <map>

namespace NS_NOMORE {

class BodyNode;
struct HeadNodeColored;
struct HeadNodeColoredChoicePoint;
struct BodyNodeColored;

class BodyLAHeuristicBase;

//! Lookahead operator that works on uncolored body nodes only.
/*! 
 *  Every uncolored body node is checked with Color::plus and Color::minus. 
 *  If assigning one color results in a conflict the node is colored to
 *  the opposite color.
 * 
 *  Additionally this operator produces information that can be used as
 *  heuristic value. For each node that can be colored either Color::plus
 *  or Color::minus this operator fires an event of type 
 *  BodyLookahead::EventType. The value member of this event type contains 
 *  counters that store the number of head nodes colored because of the 
 *  coloring of the body node.
 *  
 *  The algorithm looks like the following code fragment:
 *  \code
 *    check-queue contains all uncolored body nodes;
 *    while(check-queue not empty) {
 *      node = check-queue.pop();
 *      if (propagation with node colored to + leads to conflict) {
 *          assign(node, -);
 *      }
 *      else if (propagation with node colored to - leads to conflict) {
 *          assign(node, +);
 *      }
 *      else { 
 *        store heuristic value for node   
 *      }
 *    }
 *  \endcode
 *
 *  Technically the operator does not check every uncolored body node but
 *  only those nodes that don't get a color as a result of propagating other 
 *  nodes. That is if assigning (and propagating) color c1 to node n1 leads 
 *  to the coloring of n2 to the color c1 and n1 is supported then n2 is 
 *  removed from the check-queue. The restriction that n1 must be supported 
 *  is needed in order to get better heuristic values for supported-driven 
 *  choice-operators. */
class BodyLookahead : public Lookahead {
public:
  //! Typedef for the iterator for uncolored body nodes.
  typedef SkipIterator<OnlyNodesOfType<BodyNode*, NodeType::body_node>, Graph::UncoloredNodeIterator> iterator_type;

  //! Begin iterator of the uncolored body node set.
  /*! 
   *  \return An iterator containing the first uncolored body node of the 
   *          graph. */
  iterator_type begin() const { 
    return iterator_type(getGraph().uncoloredNodesBegin(), getGraph().uncoloredNodesEnd());
  }

  //! End iterator of the uncolored body node set.
  /*! 
   *  \return An empty iterator which defines the end of the iteration throw 
   *          the uncolored body nodes. */
  iterator_type end() const { 
    return iterator_type(getGraph().uncoloredNodesEnd(), getGraph().uncoloredNodesEnd());
  }

  //! Type that stores heuristic information for one lookahead step.
  /*!
   *  This informations are send to the heuristic. */
  struct HeuristicValueType {
    //! Constructs the object and sets the variables.
    /*!
     *  \param v1 The first value used to count the head colorings if a body 
     *            node is colored plus.
     *  \param v2 The second value used to count the head colorings if a body 
     *            node is colored minus.*/
    HeuristicValueType(long v1 = 0, long v2 = 0) {
      if (v1 > v2) {
        max_ = v1;
        min_ = v2;
      }
      else {
        max_ = v2;
        min_ = v1;
      }
    }

    //! Returns true if the current node is "better" as the values of the other node.
    /*!
     *  \param other The heuristic values of another node. 
     *  \return True if the other is not "better" than this node else false. */
    bool operator>(const HeuristicValueType& other) const {
      return max_ > other.max_ || (max_ == other.max_ && min_ > other.min_);
    }

    //! The maximum value set in constructor.
    long max_;

    //! The minimum value set in constructor.
    long min_;
  };
  
  //! Initializes the lookahead.
  /*! 
   *  \param grp The graph this lookahead should work on. */ 
  explicit BodyLookahead(Graph& grp);

  //! Handles a HeadNodeColored event fired.
  /*!
   *  \param e The raised event. */
  void handle(const HeadNodeColored& e);

  //! Handles a BodyNodeColored event fired.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColored& e);

  //! Returns the name of the operator.
  /*!
   *  \return Returns the name of the propagator. */
  const char* getName() const; 

  //! Sets the heuristic which receives the heuristic values for a node.
  /*!
   *  \param h The heuristic listen on the lookahead. */
  virtual void setHeuristic(Heuristic& h);

  //! Starts the lookahead for one node.
  /*!
   *  \param n The node which should be used for look ahead. 
   *  \return True if any conflict is found else false. */
  bool lookahead(BodyNode* n);

protected:
  //! Fires an event of type BodyLookahead::EventType.
  void storeHeuristic();

private:  
  // typedefs 
  typedef std::map<long, char> OverlayMap;
  typedef OverlayMap::iterator OverlayIterator;

  // instance methods 
  bool isOverlayed(const OverlayIterator& it, Color::ColorValue c) {
    return it != overlayMap_.end() && (it->second & c) != 0;
  }
  inline void updateOverlayMap(Node& node, char color);

  //! implements the body-lookahead
  virtual void execute();    

  // instance variables
  Color::ColorValue currentColor_;

  //! The heuristics counter. (Only heads that have one-successors)
  long counterPlus_;
  long counterMinus_;

  Node* currentNode_;
  BodyLAHeuristicBase* heuristic_;
  OverlayMap overlayMap_;
};

//! Defines the heuristic for the body lookahead informations.
class BodyLAHeuristicBase : public Heuristic {
public:
  //! Typedef for the value type of heuristic values (pair of two longs).
  typedef BodyLookahead::HeuristicValueType HeuristicValueType;

  //! Adds a heuristic value to the heuristic table.
  /*!
   *  \param n The node to set the heuristic values
   *  \param v The heuristic values for the node n.*/
  virtual void addHeuristicValue(Node* n, const HeuristicValueType& v) = 0;
};

}

#endif
