/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef NOMORE_IMPROVED_UNFOUNDED_SET_H
#define NOMORE_IMPROVED_UNFOUNDED_SET_H

#include <operators/default_local_propagator.h>
#include <color.h>
#include <head_node.h>
#include <deque>
#include "../src/detail/functors.h"
#include <set>
#include <map>
#include <queue>

namespace NS_NOMORE {

class HeadNode;
class BodyNode;
struct BodyNodeColored;
struct BodyNodeColoredChoicePoint;
struct HeadNodeColoredChoicePoint;
struct GraphConstructionFinished;

//! The implementation of a local propagation and an improved unfounded set operator.
/*! 
 *  This propagator uses the default local propagation implemented in the 
 *  class DefaultLocalPropagator. This operation is extended by the unfounded 
 *  set operator by Martin Gebser. The operator is loops motivated and searchs 
 *  a set of atoms containing a loop which can be propagated to an unfounded 
 *  set.
 *
 *  <b>Trigger:</b> The unfounded set algorithm operates on head node of 
 *                  minus colored body predecessors.
 *  <b>Init:</b> 
 *    - <i>Set</i>:      set to one head node h of a minus colored body 
 *                       predecessor which has no external bodies -- if set 
 *                       is after computation filled the node containing to a 
 *                       unfounded set.
 *    - <i>External</i>: set to body predecessors of h which are uncolored
 *    - <i>Sink</i>:     empty -- set of probably justifying head nodes
 *    - <i>Source</i>:   empty -- set of probably justifying body nodes
 *    - <i>scc</i>:      component number of h -- scc(h)
 *  
 *  <b>Algorithm:</b>
 *    \code
 *    while (External not empty) {
 *      b = External.pop();
 *      if (there is a in 0-preds of b such that scc(a) = scc and color(a) != Color::plus and a not in Sink) {
 *        J.set(body preds b of a s.t. color(b) != Color::minus and (scc(b) != scc or b in Source))
 *        if {J is empty) {
 *          Set.add(a)
 *          External.remove(0-succs of head node a)
 *          External.add(preds b of a where color(b) != Color::minus and 0-preds of b contains no head nodes from Set)
 *        } else {
 *          Sink.add(a)
 *          External.add(b)
 *        }
 *      } else {
 *        Source.add(b)
 *        R.set(0-preds of b which part of the set Set)
 *        while (R not empty) {
 *          Set.remove(R)
 *          Sink.add(R)
 *          J.set(all 0-succs bodies of head nodes of R which have no more 0-preds useful for unfounded sets)
 *          Source.push(J)
 *          R.set(0-preds of body nodes in J which part of the set Set)
 *        }
 *        External.set(body nodes which are not (probably) justified by some atom in the set Set)
 *      }
 *    }
 *    \endcode 
 *  The resulting set Set can be colored with Color::minus. */
class ImprovedUnfoundedSet : public DefaultLocalPropagator {
public:
  //! Initializes the propagator.
  /*! 
   *  \param grp The graph this propagator should work on. */ 
  explicit ImprovedUnfoundedSet(Graph& grp);

	//! Propagates using the local propagation and the improved unfounded set operator.
	/*!
   *  \return The value true is returned if the propagator colored one node. 
   *          If no node can be colored the method returns false.
	 *  \note If the graph does not contain any non-trivial strong components 
   *        (i.e. the propgram is tight) propagateFull is equivalent to 
   *        propagateLocal since tight programs can never contain loops which 
   *        are unfounded. */
  virtual bool propagateFull();

	//! Resets all members of the propagator.
  virtual void reset();

  //! Handles the GraphConstructionFinished event.
  /*! 
   *  Checks for tightness and enables or disables the unfounded set search.
   *
   *  \param e The raised event. */
  void handle(const GraphConstructionFinished& e);
  
  //! Handles a BodyNodeColored event fired.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColored& e);
  
  //! Handles a BodyNodeColoredChoicePoint event.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColoredChoicePoint& e);

  //! Returns the name of the operator.
  /*!
   *  \return Returns the name of the propagator. */
  const char* getName() const;
  
  //! Returns the state of the unfounded set finder.
  /*!
   *  \return Returns true if the propagator has disabled itself because the 
   *          graph does not contain any non-trivial strong components. */
  bool isDisabled() const;

private:
  // typedefs
  struct LessComponentNumber {
    bool operator()(const Node* n1, const Node* n2) const {
      return n1->getComponentNumber() < n2->getComponentNumber();
    }
  };

  typedef std::deque<BodyNode*> ExternalList;
  typedef std::priority_queue<HeadNode*, std::vector<HeadNode*>, LessComponentNumber> TodoList;
  typedef std::set<HeadNode*, DETAIL::LessDerefId> HeadSet;
  typedef std::set<BodyNode*, DETAIL::LessDerefId> BodySet;  
  typedef std::vector<char> NodeIndex;

  // instance variables
  bool disabled_;
  int scc_;
  int activeComponent_;
  NodeIndex source_;
  TodoList todo_;
  ExternalList external_;
  HeadSet set_;

  // instance methods
  void color(const HeadSet& unfoundedSet);
  bool findUnfoundedSet(HeadNode* head);
  void checkUnfoundedSet();  
  void updateTodoList(BodyNode* body);

  void updateExternalList(HeadNode *a);
  void updateCompleteExternal();

  bool inSink(HeadNode *h);
  bool inSource(BodyNode *b);
  bool inSet(HeadNode *h);
  bool inBodySet(const BodySet& s, BodyNode* b);

  /* returns an head a of b->zeroPredecessor with property
   *   (1) same scc AND
   *   (2) a is not plus AND
   *   (3) a is not in sink_ */
  HeadNode* selectPredOf(BodyNode *b);   

  /* returns number of a->predecessors with property
   *   (1) not same scc OR
   *   (2) b in source_ */
  int probJustifiedBodies(HeadNode *a);  

  // returns body->zeroPredecessors intersection set_ != empty set
  bool circDepended(BodyNode* b);

  /* finds all heads a with property
   *   (1) a in set_
   *   (2) a in b->successors 
   * adds all heads a to sink_ */
  void getRemovedAtoms(BodyNode *b, HeadSet &remove);

  /* finds all heads a with property
   *   (1) a in set_
   *   (2) all a->predecessors in justified
   * adds all heads a to sink_ */
  void getRemovedAtoms(const BodySet& justified, HeadSet &remove);

  /* generates the justified bodies b with the properties
   *    (1) b is zeroSuccessor of head node in remove
   *    (2) same scc
   *    (3) not minus
   *    (4) no a in b->zeropredecessors with the properties
   *             (1) same scc
   *             (2) not plus
   *             (3) not in sink_
   */
  void genJustifiedBodies(const HeadSet& remove, BodySet& justified);

  // clears all data but does not call Base::reset
  void clearInternal();   
  void clearSource();  

  inline void addToSource(Node* n);
};

}

#endif
