/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_STATIC_HEURISTIC_H
#define LIBNOMORE_STATIC_HEURISTIC_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <heuristic.h>
#include <graph.h>
#include <head_node.h>
#include <body_node.h>
#include <cassert>
#include <algorithm>

namespace NS_NOMORE {

//! Base class for all static heuristics.
/*!
 *  A static heuristic evaluates nodes only once and from then on selects the 
 *  node based on their heuristic value computate directly after graph 
 *  consruction.
 * 
 *  I.e. a static heuristic never reevaluates the heuristic value of nodes 
 *  once a value was set.
 *
 *  Nodes are valuated when the construction of the graph is finished and the 
 *  graph does not change its structure.
 *
 *  \par Template parameter: \n
 *  - HeuStrat: The policy (aka strategy) to be used to evaluate nodes and to 
 *    compare two heuristic values.
 *  .  
 *  HeuStrat must provide the following types/typedefs:
 *  - ValueType: The type of the heuristic value used by this heuristic (i.e. 
 *    int) 
 *  - ComparatorType: A function-type that accepts two values of type 
 *    ValueType and returns true if the first value is "better" than the 
 *    second.
 *  - EvaluatorType: A function-type that accepts a node. The EvaluatorType 
 *    shall return a heuristic value of type ValueType for each node for which 
 *    it is called. */
template <class HeuStrat>
class StaticHeuristic : public Heuristic {
public:
  //! The type of the heuristic value.
  typedef typename HeuStrat::ValueType ValueType;

  //! The type of the binary predicate to be used for value comparisons.
  typedef typename HeuStrat::ComparatorType ComparatorType;
  
  //! The type of the unary predicate to be used for evaluation of nodes.
  typedef typename HeuStrat::EvaluatorType EvaluatorType;

  //! The unary predicate to be used for generating a Choice out of a node.
  typedef typename HeuStrat::ChoiceCreatorType CreatorType;
  
  //! Creates a new static heuristics object.
  /*! 
   *  \param c A binary predicate used to compare two values of type 
   *           ValueType.
   *  \param e An unary predicate used to valuate nodes. */
  StaticHeuristic(const ComparatorType& c = ComparatorType(), const EvaluatorType& e = EvaluatorType())
    : comparator_(c)
    , evaluator_(e) {
  }
  
  //! Event handler for the GraphConstructionFinished event.
  /*!
   *  When receiving this event the heuristic evaluates all uncolored nodes 
   +  of the graph.
   *
   *  \param e The raised event.
   *  \pre The graph set using setGraph and the Graph contained in the 
   *       received event are equal. */
  void handle(const GraphConstructionFinished& e) {
    assert(&getGraph() == e.theGraph_);
    
    Graph::UncoloredNodeIterator it = getGraph().uncoloredNodesBegin();
    for (; it != it.end(); ++it) {
      nodeValues_.push_back(NodeValuePair(*it, evaluator_(*it, getGraph().getTightnessFactor(*it))));
    }

    std::sort(nodeValues_.begin(), nodeValues_.end(), comparator_);
  }

  //! Returns the name of the heuristic.
  /*!
   *  \return Returns the name of the heuristics used for example for the 
   *          user interface. */
  const char* getName() const {
    return NS_NOMORE::getName<StaticHeuristic>(0);
  }

private:  
  // instance methods
  //! Register at the graph.
  /*!
   *  Registers this as handler for events of type GraphConstructionFinished. */
  void doSetGraph() {
    // needed because this heuristic makes use of the tightness factor
    getGraph().checkTightness();
    getGraph().getEventManager().getChannelFor(event::Event<GraphConstructionFinished>()).connect(*this);
  }

  //! Selecting the best node wrt the heuristic values satisfing the constraint.
  /*! 
   *  \param c The constraint to be satisfying by the selected nodes. 
   *  \return Returns a "best" Choice satisfying the constraint. The choice 
   *          contains the node and its preferred and alternative. */
  Choice doSelectNode(const Constraint& c) {
    typename NodeValues::iterator end = nodeValues_.end(); 
    for (typename NodeValues::iterator it = nodeValues_.begin(); it != end; ++it) {
      if (c.isSatisfiedBy(*it->first))
        return choiceCreator_.createChoice(it->first);
    }
    return Choice();
  }

  // typedefs
  typedef std::pair<Node*, ValueType> NodeValuePair;
  typedef std::vector<NodeValuePair> NodeValues;

  // instance variables
  struct Comp {
    Comp(const ComparatorType& c)
      : cmp_(c) {
    }
    
    bool operator()(const NodeValuePair& p1, const NodeValuePair& p2) const {
      return cmp_(p1.second, p2.second);
    }
    ComparatorType cmp_;
  } comparator_;
  
  EvaluatorType evaluator_;  
  NodeValues nodeValues_;
  CreatorType choiceCreator_;
};

}
#endif
