// common implementation for hash_set and hash_map
// Copyright (c) Benjamin Kaufmann
// Adapted from SGI's hashtable (see copyright notice at end of file)
#ifndef UTIL_HASH_HELPER_H_INCLUDED
#define UTIL_HASH_HELPER_H_INCLUDED
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <stddef.h>
#include <algorithm>
#include <utility>
#include <functional>
#include <vector>
#include <iterator>

// undef this macro if your standard lib does not contain
// the std::iterator class.
#define HAS_STD_ITERATOR
namespace util { namespace detail {

// Note: assumes long is at least 32 bits.
enum { num_primes = 28 };

static const size_t prime_list[num_primes] =
{
  53u,         97u,         193u,       389u,       769u,
  1543u,       3079u,       6151u,      12289u,     24593u,
  49157u,      98317u,      196613u,    393241u,    786433u,
  1572869u,    3145739u,    6291469u,   12582917u,  25165843u,
  50331653u,   100663319u,  201326611u, 402653189u, 805306457u,
  1610612741u, 3221225473u, 4294967291u
};

inline size_t next_prime(size_t n)
{
	const size_t* first = prime_list;
	const size_t* last = prime_list + (int)num_primes;
	const size_t* pos = std::lower_bound(first, last, n);
	return pos == last ? *(last - 1) : *pos;
}

template <class V>
struct hash_node
{
	typedef V value_type;
	hash_node(const value_type& v, hash_node* next)
		: value_(v)
		, next_(next)
	{}
	value_type value_;
	hash_node* next_;
};

template <class K>
struct Identity
{
	const K& operator()(const K& n) const
	{
		return n;
	}
};

template <class V>
struct SelectFirst
{
	typedef typename V::first_type key_type;
	key_type& operator()(const V& n) const
	{
		return n.first;
	}
};

template <class Key, class Value, class HashPred, class EqPred, class KeyExtractor>
class hash_common;

template <class K, class V, class H, class E, class EE>
class hash_common_const_iterator;

template <class K, class V, class H, class E, class EE>
class hash_common_iterator
#ifdef HAS_STD_ITERATOR
	: public std::iterator<std::forward_iterator_tag, V, ptrdiff_t>
#endif
{
	typedef hash_common<K, V, H, E, EE> hash_common_type;
	typedef hash_node<V> node_type;
	friend class hash_common_const_iterator<K, V, H, E, EE>;
public:
#ifndef HAS_STD_ITERATOR
	typedef std::forward_iterator_tag iterator_category;
	typedef V value_type;
	typedef ptrdiff_t difference_type;
#endif
	typedef size_t size_type;
	typedef V& reference;
	typedef V* pointer;
	hash_common_iterator() {}
	hash_common_iterator(node_type* n, hash_common_type* s)
		: node_(n)
		, set_(s)
	{}

	reference operator*() const { return node_->value_;}
	pointer operator->() const { return &**this;}

	hash_common_iterator& operator++()
	{
		node_type* save = node_;
		node_ = node_->next_;
		if (!node_)
		{
			size_type currBucket = set_->hash(set_->key_(save->value_));
			while (!node_ && ++currBucket < set_->buckets_.size())
			{
				node_ = set_->buckets_[currBucket];
			}
		}
		return *this;
	}
	hash_common_iterator operator++(int)
	{
		hash_common_iterator temp(*this);
		++*this;
		return temp;
	}
	bool operator==(const hash_common_iterator& rhs) const
	{
		return node_ == rhs.node_;
	}
	bool operator!=(const hash_common_iterator& rhs) const
    {
		return node_ != rhs.node_;
	}
private:
	node_type* node_;
	hash_common_type* set_;
};

template <class K, class V, class H, class E, class EE>
class hash_common_const_iterator
#ifdef HAS_STD_ITERATOR
	: public std::iterator<std::forward_iterator_tag, V, ptrdiff_t>
#endif
{
	typedef hash_common<K, V, H, E, EE> hash_common_type;
	typedef hash_node<V> node_type;
	typedef hash_common_iterator<K, V, H, E, EE> iterator;
public:
#ifndef HAS_STD_ITERATOR
	typedef std::forward_iterator_tag iterator_category;
	typedef V value_type;
	typedef ptrdiff_t difference_type;
#endif
	typedef size_t size_type;
	typedef const V& reference;
	typedef const V* pointer;
	hash_common_const_iterator() {}
	hash_common_const_iterator(const node_type* n, const hash_common_type* s)
		: node_(n)
		, set_(s)
	{}
	hash_common_const_iterator(iterator it)
		: node_(it.node_)
		, set_(it.set_)
	{}

	reference operator*() const { return node_->value_;}
	pointer operator->() const { return &**this;}
	hash_common_const_iterator& operator++()
	{
		const node_type* save = node_;
		node_ = node_->next_;
		if (!node_)
		{
			size_type currBucket = set_->hash(set_->key_(save->value_));
			while (!node_ && ++currBucket < set_->buckets_.size())
			{
				node_ = set_->buckets_[currBucket];
			}
		}
		return *this;
	}
	hash_common_const_iterator operator++(int)
	{
		hash_common_const_iterator temp(*this);
		++*this;
		return temp;
	}
	bool operator==(const hash_common_const_iterator& rhs) const
	{
		return node_ == rhs.node_;
	}
	bool operator!=(const hash_common_const_iterator& rhs) const
    {
		return node_ != rhs.node_;
	}
private:
	const node_type* node_;
	const hash_common_type* set_;
};



template <class Key, class Value, class HashPred, class EqPred, class KeyExtractor>
class hash_common
{
	typedef hash_node<Value> node_type;
	friend class detail::hash_common_iterator<Key, Value, HashPred, EqPred, KeyExtractor>;
	friend class detail::hash_common_const_iterator<Key, Value, HashPred, EqPred, KeyExtractor>;
public:
	typedef Key key_type;
    typedef Value value_type;
	typedef HashPred hasher_type;
	typedef EqPred key_compare;
	typedef size_t	size_type;
	typedef ptrdiff_t difference_type;
	typedef value_type* pointer;
	typedef const value_type* const_pointer;
	typedef value_type& reference;
	typedef const value_type& const_reference;
	typedef hash_common_iterator<Key, Value, HashPred, EqPred, KeyExtractor> iterator;
	typedef hash_common_const_iterator<Key, Value, HashPred, EqPred, KeyExtractor> const_iterator;

	hash_common(size_t n, float lf, const HashPred& h, const EqPred& e, const KeyExtractor& k)
		: hasher_(h)
		, key_comparer_(e)
		, elements_(0)
		, loadFactor_(lf)
		, key_(k)
	{
		buckets_.resize(detail::next_prime(n), static_cast<node_type*>(0));
	}

	hash_common(const hash_common& other)
		: hasher_(other.hasher_)
		, key_comparer_(other.key_comparer_)
		, elements_(other.elements_)
		, loadFactor_(other.loadFactor_)
		, key_(other.key_)
	{
		buckets_.resize(other.buckets_.size(), static_cast<node_type*>(0));
		try {
		for (size_type i = 0; i != other.buckets_.size(); ++i)
		{
			if (const node_type* curr = other.buckets_[i])
			{
				node_type* newNode = new node_type(curr->value_, 0);
				buckets_[i] = newNode;
				for (node_type* next = curr->next_; next; curr = next, next = curr->next_)
				{
					newNode->next_ = new node_type(next->value_, 0);
					newNode = newNode->next_;
				}
			}
		}
		}
		catch(...)
		{
			clear();
			throw;
		}
	}

	hash_common& operator=(const hash_common& other)
	{
		if (this != &other)
		{
			hash_common temp(other);
			temp.swap(*this);
		}
		return *this;
	}

	size_type count(const key_type& k) const
	{
		return find(k) == end() ? 0 : 1;
	}
	void swap(hash_common& other)
	{
		std::swap(hasher_, other.hasher_);
		std::swap(key_comparer_, other.key_comparer_);
		buckets_.swap(other.buckets_);
		std::swap(elements_, other.elements_);
		std::swap(loadFactor_, other.loadFactor_);
		std::swap(key_, other.key_);
	}

	std::pair<iterator, bool> insert(const value_type& x)
	{
		resize();
		size_type h = hash(key_(x));
		node_type* first = buckets_[h];
		for (node_type* run = first; run; run = run->next_)
		{
			if (key_comparer_(key_(run->value_), key_(x)))
				return std::pair<iterator, bool>(iterator(run, this), false);
		}

		node_type* newNode = new node_type(x, first);
		buckets_[h] = newNode;
		++elements_;
		return std::pair<iterator, bool>(iterator(newNode, this), true);
	}

	template <class Iter>
	void insert(Iter first, Iter last)
	{
		while (first != last)
		{
			insert(*first);
			++first;
		}
	}

	const_iterator find(const key_type& key) const
	{
		for (node_type* c = buckets_[hash(key)]; c; c = c->next_)
		{
			if (key_comparer_(key_(c->value_), key))
				return const_iterator(c, this);
		}
		return end();
	}

	iterator find(const key_type& key)
	{
		for (node_type* c = buckets_[hash(key)]; c; c = c->next_)
		{
			if (key_comparer_(key_(c->value_), key))
				return iterator(c, this);
		}
		return end();
	}

	size_type erase(const key_type& key)
	{
		size_type h = hash(key);
		if (node_type* f = buckets_[h])
		{
			if (key_comparer_(key_(f->value_), key))
			{
				buckets_[h] = f->next_;
				delete f;
				--elements_;
				return 1;
			}
			node_type* curr = f;
			node_type* next = curr->next_;
			while (next)
			{
				if (key_comparer_(key_(next->value_), key))
				{
					curr->next_ = next->next_;
					delete next;
					--elements_;
					return 1;
				}
				curr = next;
				next = curr->next_;
			}
		}
		return 0;
	}

	void erase(const_iterator it)
	{
		erase(key_(*it));
	}

	void erase(const_iterator first, const_iterator last)
	{
		while (first != last)
		{
			erase(first++);
		}
	}

	void clear()
	{
		deleteAll(buckets_);
		elements_ = 0;
	}

	iterator begin()
	{
		for (size_type i = 0; i != buckets_.size(); ++i)
		{
			if (buckets_[i])
				return iterator(buckets_[i], this);
		}
		return end();
	}

	const_iterator begin() const
	{
		for (size_type i = 0; i != buckets_.size(); ++i)
		{
			if (buckets_[i])
				return const_iterator(buckets_[i], this);
		}
		return end();
	}

	iterator end()
	{
		return iterator(static_cast<node_type*>(0), this);
	}

	const_iterator end() const
	{
		return const_iterator(static_cast<const node_type*>(0), this);
	}

	size_type size() const
	{
		return elements_;
	}
	size_type max_size() const
	{
		return size_type(-1);
	}
	bool empty() const
	{
		return elements_ == 0;
	}
	EqPred key_comp() const
	{
		return key_comparer_;
	}
private:
	HashPred hasher_;
	EqPred key_comparer_;
	size_t	elements_;
	float		loadFactor_;
	std::vector<node_type*> buckets_;
	KeyExtractor key_;

	size_type hash(const key_type& v, size_type size) const
	{
		return hasher_(v) % size;
	}

	size_type hash(const key_type& v) const
	{
		return hash(v, buckets_.size());
	}

	void deleteAll(std::vector<node_type*>& v)
	{
		for (size_t i = 0; i != v.size(); ++i)
		{
			while (v[i])
			{
				node_type* run = v[i]->next_;
				delete v[i];
				v[i] = run;
			}
			v[i] = 0;
		}
	}

	void resize()
	{
		if (elements_ > loadFactor_ * buckets_.size())
		{
			size_t newSize = detail::next_prime(buckets_.size() + 1);
			if (newSize > buckets_.size())
			{
				std::vector<node_type*> temp(newSize, static_cast<node_type*>(0));
				try
				{
					size_type oldSize = buckets_.size();
					for (size_t i = 0; i != oldSize; ++i)
					{
						node_type* first = buckets_[i];
						while (first)
						{
							size_t rehash = hash(key_(first->value_), newSize);
							buckets_[i] = first->next_;
							first->next_ = temp[rehash];
							temp[rehash] = first;
							first = buckets_[i];
						}
					}
					buckets_.swap(temp);
				}
				catch(...)
				{
					deleteAll(temp);
					throw;
				}
			}
		}
	}
};


} }

/*
 * Copyright (c) 1996-1998
 * Silicon Graphics Computer Systems, Inc.
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Silicon Graphics makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.  Hewlett-Packard Company makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 */

#ifdef HAS_STD_ITERATOR
#undef HAS_STD_ITERATOR
#endif
#endif
