/***************************************************************************
                          cdetoperator.h  -  description
                             -------------------
    begin                : Mon Aug 23 2004
    copyright            : (C) 2004 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef CDETOPERATOR_H
#define CDETOPERATOR_H

#include "coperator.h"

namespace NS_NOMORE {

/*! \class CDetOperator coperator.h
		\brief Interface for defining operators.

		CDetOperator is the base class for all deterministic operators. All these operators must be
    implement the Call() methode.
*/
class CDetOperator : virtual public COperator {
	//! Output operator
	friend std::ostream& operator<<(std::ostream& os, CDetOperator& op);

public:
  //! constructor
	CDetOperator();

  //! destructor
	virtual ~CDetOperator();

  //! starts the operator computation
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack) = 0;

  //! Equality operator work on operator ids
  virtual bool operator==(CDetOperator& op);

  //! Inequality operator work on operator ids
  virtual bool operator!=(CDetOperator& op);
};



/*! \class CDetOperatorNone coperator.h
		\brief Operator does not change anything.
*/
class CDetOperatorNone : public CDetOperator {
public:
  //! constructor
	CDetOperatorNone();

  //! destructor
	virtual ~CDetOperatorNone();

  //! do nothing
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
  virtual TType GetType();

//  virtual bool Conflict(TType op);
};



/*! \class CDetOperatorP coperator.h
		\brief Deterministic operator P defined in paper.

		CDetOperatorP computes the deterministic consequences of the colored RDG.
*/
class CDetOperatorP : public CDetOperator {
public:
  //! constructor
	CDetOperatorP();

  //! destructor
	virtual ~CDetOperatorP();

  //! starts the operatorP which colors all supported and unblocked plus and all unsupported and all blocked minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();

};



/*! \class CDetOperatorPs coperator.h
		\brief Deterministic operator P. Implements P*.

		CDetOperatorPs computes the deterministic consequences of the colored RDG.
*/
class CDetOperatorPs : public CDetOperator {
public:
  //! constructor
	CDetOperatorPs();

  //! destructor
	virtual ~CDetOperatorPs();

  //! starts the operatorP which colors all supported and unblocked plus and all unsupported and all blocked minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};


/*! \class CDetOperatorPsl coperator.h
		\brief linear Deterministic operator P defined in paper.

		CDetOperatorPsl computes the deterministic consequences of the colored RDG.
*/
class CDetOperatorPsl : public CDetOperator {
public:
  //! constructor
	CDetOperatorPsl();

  //! destructor
	virtual ~CDetOperatorPsl();

  //! starts the operatorP which colors all supported and unblocked plus and all unsupported and all blocked minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();

};


/*! \class CDetOperatorU coperator.h
		\brief Deterministic operator U defined in paper.

		CDetOperatorU computes a maximal support graph and colors all node not in this graph with minus.
*/
class CDetOperatorU : public CDetOperator {
public:
  //! constructor
	CDetOperatorU();

  //! destructor
	virtual ~CDetOperatorU();

  //! starts the operatorU which generates the maximal support graph and colors all rules not in the maximal support graph with minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};



/*! \class CDetOperatorN coperator.h
		\brief Deterministic operator N defined in paper.

		CDetOperatorN colors all uncolored nodes with minus.
*/
class CDetOperatorN : public CDetOperator {
public:
  //! constructor
	CDetOperatorN();

  //! destructor
	virtual ~CDetOperatorN();

  //! starts the operatorN colors all uncolored nodes to minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return If an error occurs return_colorerror is returned. Othgerwise the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};



/*! \class CDetOperatorV coperator.h
		\brief Deterministic operator V defined in paper.
*/
class CDetOperatorV : public CDetOperator {
public:
  //! constructor
	CDetOperatorV();

  //! destructor
	virtual ~CDetOperatorV();

  //! starts the operatorV which generates the maximal support graph and colors all rules not in the maximal support graph with minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};


/*! \class CDetOperatorPre coperator.h
		\brief Deterministic operator Pre.

		CDetOperatorPre colors selfblocker with minus and all facts with plus.
*/
class CDetOperatorPre : public CDetOperator {
public:
  //! constructor
	CDetOperatorPre();

  //! destructor
	virtual ~CDetOperatorPre();

  //! Colors all self-blocker minus and all facts plus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not)*/
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};



/*! \class CDetOperatorPrePref coperatorpref.h
		\brief Deterministic operator P defined in paper.*/


class CDetOperatorPrePref : public CDetOperator{
public:
  //! constructor
	CDetOperatorPrePref();

  //! destructor
	virtual ~CDetOperatorPrePref();

  //! Colors all self-blocker minus and all facts plus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not)*/
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};


/*! \class CDetOperatorPPref coperator.h
		\brief Deterministic operator P defined in paper.

	CDetOperatorPPref computes the deterministic consequences of the colored RDG.
*/
class CDetOperatorPPref : public CDetOperator {
public:
  //! constructor
	CDetOperatorPPref();

  //! destructor
	virtual ~CDetOperatorPPref();

  //! starts the operatorP which colors all supported and unblocked plus and all unsupported and all blocked minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};


/*! \class CDetOperatorPsPref coperatorpref.h
		\brief Deterministic operator P. Implements P*.

		COperatorPsPref computes the deterministic consequences of the colored RDG.
*/
class CDetOperatorPsPref : public CDetOperator{
public:
  //! constructor
	CDetOperatorPsPref();

  //! destructor
	virtual ~CDetOperatorPsPref();

  //! starts the operatorP which colors all supported and unblocked plus and all unsupported and all blocked minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};


/*! \class CDetOperatorPslPref coperator.h
		\brief linear Deterministic operator P defined in paper.

		CDetOperatorPslPref computes the deterministic consequences of the colored RDG.
*/
class CDetOperatorPslPref : public CDetOperator {
public:
  //! constructor
	CDetOperatorPslPref();

  //! destructor
	virtual ~CDetOperatorPslPref();

  //! starts the operatorP which colors all supported and unblocked plus and all unsupported and all blocked minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();

};

/*! \class CDetOperatorUPref coperatorpref.h
		\brief Deterministic operator U defined in paper.

		COperatorUPref computes a maximal support graph and colors all node not in this graph with minus.
*/
class CDetOperatorUPref : public CDetOperator{
public:
  //! constructor
	CDetOperatorUPref();

  //! destructor
	virtual ~CDetOperatorUPref();

  //! starts the operatorU which generates the maximal support graph and colors all rules not in the maximal support graph with minus
  /*! \param graph Graph on which the computation is started
      \param stack Stack on which the changes are stored
      \return if any error occured the computation the methode return return_colorerror else the methode
              returns the state of the computation (something has changed or not) */
  virtual TRetOperator Call(CGraph* graph, TStack* stack);

  //! returns an operator specific id which is used for testing of equality
	virtual TType GetType();
};

} // end of namespace NS_NOMORE

#endif
