%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
% Generate Examples for Testing                                           %
%                                                                         %
% Kathrin Konczak, Christian Anger                                        %
%                                                                         %
% last edit: Jan 2002                                                     %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- ensure_loaded(graphgenerator).
:- ensure_loaded(theorybase).
:- ensure_loaded(io).
:- ensure_loaded(library(lists)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% generate_examples/0 - interface                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

generate_examples:-
	welcome,
	graph_type(GraphType),
	parameter(GraphType,GraphCall),
	generate_graph(GraphCall,(V,I)),
	in_between,
	theory_type(TheoryType),
	theory_call(TheoryType,(V,I),TheoryCall),
	theorybase(TheoryCall,LP),
	in_between,
	encoding(Encoding),
	in_between,
	filename(File),
	lp2file(Encoding,LP,File),
	bye_bye.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% generate_examples/4 - interface                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% generate_examples(+GraphCall,+TheoryType,+Encoding,+File)
% where GraphCall is one of the following:
%    circle(N)    - N is number of nodes of circle graph
%    complete(N)  - N is number of nodes of complete graph
%    ladder(N)    - N is number of runges of ladder graph
%    kladder(N,K) - N is number of runges and k number of runges missing
%    simplex(N)   - N is number of nodes of circle graph
% TheroyType is on of the following
%    coloring1 - coloring I theory
%    coloring2 - coloring II theory
%    hamilton  - hamilton theory
%    indset    - independent set theory
% Encoding is one of the following
%    dlv  - encoding for dlv
%    sm   - encoding for smodels
%    quip - encoding for quip
% File is a file name
%
%
% examples calls:
%
% generate_examples(complete(4),hamltion,sm,'hamk4.lp')
% generate_examples(ladder(10),indset,sm,'ind_lad_10.lp')
%
generate_examples(GraphCall,TheoryType,Encoding,File):-
	generate_graph(GraphCall,(V,I)),
	theory_call(TheoryType,(V,I),TheoryCall),
	theorybase(TheoryCall,LP),
	lp2file(Encoding,LP,File).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% implementations                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

welcome:-
	write('**************************************************'),nl,
	write('*             Example Generator V1.0             *'),nl,
        write('**************************************************'),nl,
	nl.

graph_type(Graph):-
	write('Please choose a graph type:  (Please finish your  '),nl,
	write('                              input with <ENTER>) '),nl,
	write('(1) Circle                                        '),nl,
	write('(2) Complete                                      '),nl,
	write('(3) Ladder                                        '),nl,
	write('(4) K-Ladder                                      '),nl,
	write('(5) Simplex                                       '),nl,
	nl,
	write('Please choose one of the above (1-5): '),
	read_string("\n",_,Input),nl,
	(
	    member(Input,["1","2","3","4","5"]) ->
	    (
		Input = "1" ->
		(
		    Graph = circle
		);
		(
		    Input = "2" ->
		    (
			Graph = complete
		    );
		    (
			Input = "3" ->
			(
			    Graph = ladder
			);
			(
			    Input = "4" ->
			    (
				Graph = kladder
			    );
			    (
				Graph = simplex
			    )
			)
		    )
		)
	    );
	    (
		write('!! Sorry, your input was incorrect !! '),nl,nl,
		graph_type(Graph)
	    )
	).
		    

parameter(circle,circle(N)):-
	write('How many nodes do you want the circle graph to have?'),nl,
	write('(Please Enter a number followd by <ENTER>)    '),nl,
	parameter(N).
		
parameter(complete,complete(N)):-	  
	write('How many nodes do you want the complete graph to have?'),nl,
	write('(Please Enter a number followd by <ENTER>)    '),nl,
	parameter(N).

parameter(ladder,ladder(N)):-	  
	write('How many runges do you want the ladder graph to have?'),nl,
	write('(Please Enter a number followd by <ENTER>)    '),nl,
	parameter(N).

parameter(kladder,kladder(N,K)):-	  
	write('How many runges do you want the kladder graph to have?'),nl,
	write('(Please Enter a number followd by <ENTER>)    '),nl,
	parameter(N1),
	write('How many runges do you want missing between two runges?'),nl,
	write('(Please Enter a number followd by <ENTER>)    '),nl,
	parameter(K1),
	Test1 is N1-1,
	Test2 is K1+1,
	mod(Test1,Test2,Test3),
	(
	    Test3 = 0 ->
	    (
		N = N1, K = K1
	    );
	    (
		write('!! Sorry, does not give a whole kladder,'),
		write('use different numbers  !! '),nl,nl,
		parameter(kladder,kladder(N,K))
	    )
	).


parameter(simplex,simplex(N)):-	  
	write('How many nodes do you want the simplex graph to have?'),nl,
	write('(Please Enter a number followd by <ENTER>)    '),nl,
	parameter(N).

parameter(N):-
	read_string("\n",_,Input),nl,
	(
	    (atom_string(Atom,Input),
	      integer_atom(Integer,Atom),
	      Integer > 0) ->
	    (
		N = Integer
	    );
	    (
		write('!! Sorry, your input was incorrect !! '),nl,nl,
	        parameter(N)
	    )
	).
		

theory_type(Theory):-
	write('Please choose a theory type: (Please finish your  '),nl,
	write('                              input with <ENTER>) '),nl,
	write('(1) Graph Coloring I                              '),nl,
	write('(2) Graph Coloring II                             '),nl,
	write('(3) Hamiltonian Cycles                            '),nl,
	write('(4) Independet Sets                               '),nl,
	nl,
	write('Please choose one of the above (1-4): '),
	read_string("\n",_,Input),nl,
	(
	    member(Input,["1","2","3","4"]) ->
	    (
		Input = "1" ->
		(
		    Theory = coloring1
		);
		(
		    Input = "2" ->
		    (
			Theory = coloring2
		    );
		    (
			Input = "3" ->
			(
			    Theory = hamilton
			);
			(
			    Theory = indset
			)
		    )
		)
	    );
	    (
		write('!! Sorry, your input was incorrect !! '),nl,nl,
		theory_type(Theory)
	    )
	).
		    
theory_call(coloring1,(V,I),coloring(3,(V,A),[green,red])):-
	succ2arc(I,A).
theory_call(coloring2,(V,I),coloring(2,(V,A),[green,red,blue])):-
	succ2arc(I,A).
theory_call(hamilton,([First|V],A),hamilton((V,A),First)).
theory_call(indset,(V,I),indset((V,A))):-
	succ2arc(I,A).

encoding(Encoding):-
	write('Please choose a encoding: (Please finish your  '),nl,
	write('                           input with <ENTER>) '),nl,
	write('(1) DLV                                        '),nl,
	write('(2) SModels                                    '),nl,
	write('(3) Quip                                       '),nl,
	nl,
	write('Please choose one of the above (1-3): '),
	read_string("\n",_,Input),nl,
	(
	    member(Input,["1","2","3"]) ->
	    (
		Input = "1" ->
		(
		    Encoding = dlv
		);
		(
		    Input = "2" ->
		    (
			Encoding = sm
		    );
		    (
			Encoding = quip

		    )
		)
	    );
	    (
		write('!! Sorry, your input was incorrect !! '),nl,nl,
		encoding(Encoding)
	    )
	).

filename(File):-
	write('Please choose a file name: (Please finish your  '),nl,
	write('                            input with <ENTER>) '),nl,
	nl,
	write('(e.g. hamk4.lp, ind_cir_30.lp ...): '),
	read_string("\n",_,File2),nl,
	(
	    existing_file(File2,[""],[],_) ->
	    (
		write('!! File exists !!   Overwrite? (y/n)'),
		read_string("\n",_,Input),nl,
		(
		    Input = "y" ->
		    (
			File = File2
		    );
		    (
			filename(File)
		    )
		)
	    );
	    (
		File = File2
	    )
	).


in_between:-
        write('**************************************************'),nl.

bye_bye:-
        write('**************************************************'),nl,
	write('    Thank you for using Example Generator V1.0    '),nl,
        write('**************************************************'),nl.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% i/o predicates                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% lp2file(Option,LP,File) 
% Option in {sm,dlv,quip}
%
lp2file(Option,LP,File):-
	int2ext(LP,Ext),
	write2file(Option,File,Ext).


%
% write2file/3
% opens and closes the stream for writing
%
write2file(Option,File,LP):-
	open(File,write,Stream),
	write_clauses(Option,Stream,LP),
	close(Stream).

%
% write_clauses/3
% write clauses in choosen syntax
%
% smodels
write_clauses(sm,_,[]).
write_clauses(sm,Stream,[H|T]):-
	write(Stream,H),
	writeln(Stream,"."),
	write_clauses(sm,Stream,T).
% dlv
write_clauses( dlv, Stream, Clauses ) :-
	write(Stream,"true."),nl(Stream),
	write_clauses_dlv(Stream,Clauses).
% quip
write_clauses( quip, Stream, Clauses ) :-
	write(Stream,"@C lp1 :="),nl,
	write_clauses_quip(Stream,Clauses),
	writeln(Stream,"@LP(@C lp1,true:-.)").

%
% write_clauses_dlv/2
%
write_clauses_dlv(_Stream, [] ) :- !.
write_clauses_dlv(Stream, [H|T]):-
	write(dlv,Stream,H),
	writeln(Stream,"."),
	write_clauses_dlv(Stream,T).

%
% write_clauses_quip/2
%
write_clauses_quip(Stream, [H] ) :-
	write(quip,Stream,H),
	writeln(Stream,".").
write_clauses_quip(Stream, [H|T]):-
	write(quip,Stream,H),
	writeln(Stream,".,"),
	write_clauses_quip(Stream,T).

%
% write/3
%
write( Option, Stream, (Head :- Body) ) :-
	(
	    atom(Head) ->
	    (
		write(Stream,Head)
	    );
	    (
		convert_term(Head,Head2),
		write(Stream,Head2)
	    )
	),
	write(Stream," :- "),
 	writeA(Option,Stream,Body).
write( Option, Stream, Fact ) :-
	(
	    atom(Fact) ->
	    (
		write(Stream,Fact)
	    );
	    (
		convert_term(Fact,Fact2),
		write(Stream,Fact2)
	    )
	),
	(
	    Option = quip ->
	    (
		write(Stream," :- ")
	    );
	    (
		Option = dlv,
		write(Stream," :- true")
	    )
	).

%
% writeA/3
%
writeA( Option, Stream, (A,B) ) :-
	writeA(Option,Stream,A),
	write(Stream,", "),
 	writeA(Option,Stream,B).
writeA( Option, Stream, not(A) ) :-
	(
	    Option = quip ->
	    (
		write(Stream,"~"),
		writeA(quip,Stream,A)
	    );
	    (
		Option = dlv,
		write(Stream,"not "),
		writeA(dlv,Stream,A)
	    )
	).
writeA( _Option, Stream, A ) :-
	not(atom(A)),
	convert_term(A,A2),
	write(Stream,A2).
writeA( _Option, Stream, A ) :-
	atom(A),
	write(Stream,A).

%
% convert_term/2
%
convert_term( A, A2 ) :-
	not(atom(A)),
	term_string(A,S),
	split_string(S,"(,) ","",S1),
	concat_string(S1,S2),
	term_string(A2,S2).

