%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
% Graph Generator                                                         %
% Christian Anger, Kathrin Konczak, Thomas Linke                          %
%                                                                         %
% last edit: Feb 21, 2001                                                 %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


:-ensure_loaded(graphs).


%
% generate_graph(+T,-G) generates a graph G=(Nodes,Arcs), where V=nodes
% and E=arcs in ArcList, of Typ T, T being one of the following
% circle(N)  : circle with N nodes
% complete(N): complete graph with N nodes
% ladder(N)  : ladder graph with 2*n nodes
% kladder(N) : ladder graph with k missing ladders
% simplex(N) : simplex graph with N levels
%

generate_graph(circle(N),G):-circle(N,G).
generate_graph(complete(N),G):-complete_graph(N,G).
generate_graph(ladder(N),G):-ladder(N,G).
generate_graph(kladder(N,K),G):-kladder(N,K,G).
generate_graph(simplex(N),G):-simplex_graph(N,G).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Circle                                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% circle(+N,+G) genrates a graph G=(Nodes,Arcs), Arcs in ArcList
% with N veriteces and arcs, that form a single circle over all nodes
%

circle(N,(V,I)):-circle(1,N,([],[]),(V,A)),
	         arc2succ(V,A,I).


circle(1,1,(V,E),([S|V],E)):-
	make_symbol(1,"v",S),!.
circle(N,N,(V,E),([S1|V],[S1-S2|E])):-
	make_symbol(N,"v",S1),
	make_symbol(1,"v",S2),!.
circle(Z,N,(V,E),([S1|V_Out],[S1-S2|E_Out])):-
	Z1 is Z+1,
	make_symbol(Z,"v",S1),
	make_symbol(Z1,"v",S2),
	circle(Z1,N,(V,E),(V_Out,E_Out)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Complete                                                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% complete_graph(+N,-G) genrates a complete Graph G=(V,E), where 
% V is the set of vertices ( N=|V| )
% and E are the edges in IncList
%
complete_graph(N,(V,I)):-
        build_vertices(N,[],V),
        build_complete_edges(V,V,[],A),
        arc2succ(V,A,I).


%
% build_complete_edges/4
%
build_complete_edges([],_,In,In).
build_complete_edges([H|T],V,In,Out):-
	build(V,H,[],L),
	build_complete_edges(T,V,In,Out1),
	append(L,Out1,Out).

%
% build/4
%
build([],_,In,In).
build([V|T],V,In,Out):-
	build(T,V,In,Out),!.
build([H|T],V,In,[V-H|Out]):-
	build(T,V,In,Out).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Ladder                                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% ladder(+N,-G) generates a ladder graph G=(V,E), with 2*N Vertices and
% E in IncList 
%
ladder(N,(V,I)):-
	N2 is N*2,
	build_vertices(N2,[],V),
	side_edges(N,V,V_Out,[],R),
	side_edges(N,V_Out,_,[],L),
	middle_edges_even(V,[],M),
	union(R,L,E1),
	union(E1,M,E),
        arc2succ(V,E,I).


%
% side_edges/5
%
side_edges(1,[_],[],In,In):-!.
side_edges(1,[_|[V2|T]],[V2|T],In,In):-!.
side_edges(N,[V1|[V2|T]],V_Out,In,[V1-V2|Out]):-
	N1 is N-1,
	side_edges(N1,[V2|T],V_Out,In,Out).


%
% middle_edges_even/3
%
middle_edges_even([],In,In):-!.
middle_edges_even([H|T],In,[L-H|Out]):-
	last(T,[],T_Out,L),
	middle_edges_uneven(T_Out,In,Out).


%
% middle_edges_uneven/3
%
middle_edges_uneven([],In,In):-!.
middle_edges_uneven([H|T],In,[H-L|Out]):-
	last(T,[],T_Out,L),
	middle_edges_even(T_Out,In,Out).


%
% last/4
%
last([L],In,In,L).
last([H|T],In,[H|Out],L):-
	last(T,In,Out,L).
last([],[],[]):-fail,!.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% k-Ladder                                                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% let N be the number of runges (Leitersprossen) 
% let k be the number of runges missing between 2 runges 
%
%   v_2(N-1)k+1       --->  v_0
%       |                    |
%   v_2(N-1)k               v_1 
%       |                    |
%      ...                  ...   
%       |                    |
%   v_2(N-1)k-k      <---   v_k
%       |                    |
%      ...                  ...   
%       |                    |
%   v_(N-1)k+1       (1)   v_(N-1)k
%
% (1) if N is even then <---
%     if N is odd  then --->
% 
% kladder(+N,+K,-Graph)
% edges in inc representation
%
kladder(N,K,(V,I)):-
    vert_left(N,K,EL),
    vert_right(N,K,ER),
    N1 is N-1,
    horizont(N1,N,K,EH),
    append(EL,ER,EV),
    append(EV,EH,Edges),
    build_nodes(N,K,V),
    arc2succ(V,Edges,I).


%
% build_nodes/3 /2
%
build_nodes(N,K,Nodes):-
    C is 2*(N-1)*K+1,
    build_nodes(C,Nodes).

build_nodes(0,[v0]).
build_nodes(C,[VC|Nodes]):-
    term_string(C,SC),
    append_strings("v",SC,NC),
    term_string(VC,NC),
    C1 is C-1,
    build_nodes(C1,Nodes).


%
% vert_right/3 /2
%
vert_right(N,K,ER):-
    C is (N-1)*K,
    vert_right(C,ER).

vert_right(0,[]).
vert_right(C,[VertC1-VertC|ER]):-
    C1 is C-1,
    term_string(C,SC),
    term_string(C1,SC1),
    append_strings("v",SC,VSC),
    append_strings("v",SC1,VSC1),
    term_string(VertC,VSC),
    term_string(VertC1,VSC1),
    vert_right(C1,ER).
    

%
% vert_left/3 /2
%
vert_left(N,K,EL):-
    Y is 2*(N-1)*K+1, X is (N-1)*K,
    vert_lefth(Y,X,EL).

vert_lefth(_Y,0,[]).
vert_lefth(Y,X,[VertC-VertC1|EL]):-
    C is Y-X, C1 is C+1,
    term_string(C,SC),
    term_string(C1,SC1),
    append_strings("v",SC,VSC),
    append_strings("v",SC1,VSC1),
    term_string(VertC,VSC),
    term_string(VertC1,VSC1),
    X1 is X-1,
    vert_lefth(Y,X1,EL).


%
% horizont/4
%
horizont(-1,_,_,[]).
horizont(Lauf,N,K,[VertC-VertC1|EH]):-
      	0 is Lauf mod 2,!,
	C is 2*(N-1)*K+1-(Lauf*K),
        C1 is Lauf*K,
        term_string(C,SC),
        term_string(C1,SC1),
        append_strings("v",SC,VSC),
        append_strings("v",SC1,VSC1),
        term_string(VertC,VSC),
        term_string(VertC1,VSC1),
        LaufN is Lauf-1,
	horizont(LaufN,N,K,EH).
horizont(Lauf,N,K,[VertC1-VertC|EH]):-
	C is 2*(N-1)*K+1-(Lauf*K),
        C1 is Lauf*K,
        term_string(C,SC),
        term_string(C1,SC1),
        append_strings("v",SC,VSC),
        append_strings("v",SC1,VSC1),
        term_string(VertC,VSC),
        term_string(VertC1,VSC1),
        LaufN is Lauf-1,
	horizont(LaufN,N,K,EH).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% other Stuff                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% build_vertices/3
%
build_vertices(0,In,In):-!.
build_vertices(N,In,Out):-
	M is N-1,
	make_symbol(N,"v",S),
	build_vertices(M,[S|In],Out).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% simplex graph                                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% simplex_graph(+N,-G) 
% constructs a simplex graph G with N runges 
%

simplex_graph(N,(V,E)):-
	construct_vertices(1,N,[],V),
	simplex_graph(1,N,[],A),
	arc2succ(V,A,E).


construct_vertices(M,N,V,V):-
	M > N, !.
construct_vertices(M,N,V_In,V_Out):-
	construct_vertices2(1,M,V_In,V1),
	M1 is M+1,
	construct_vertices(M1,N,V1,V_Out).

construct_vertices2(R,M,V,V):-
	R > M, !.
construct_vertices2(R,M,V_In,V_Out):-
	term_string(R,SR),
	term_string(M,SM),
	append_strings("v__",SM,S1),
	append_strings(S1,"_",S2),
	append_strings(S2,SR,V),
	term_string(V__M_R,V),
	R1 is R+1,
	construct_vertices2(R1,M,[V__M_R|V_In],V_Out).


simplex_graph(N1,N,E,E):-N1 > N,!.
simplex_graph(R,N,E_In,E_Out):-
	simplex_graph2(R,R,E_In,E1),
	R1 is R+1,
	simplex_graph(R1,N,E1,E_Out).


simplex_graph2(1,_R,E,E):-!.
simplex_graph2(M,R,E_In,E_Out):-
	M1 is M-1,R1 is R-1,

	term_string(R,SR),
	term_string(R1,SR1),
	term_string(M,SM),
	term_string(M1,SM1),

	append_strings("v__",SR,S1_1),
	append_strings(S1_1,"_",S2_1),
	append_strings(S2_1,SM,V__M_R),
	term_string(V1,V__M_R),
	
	append_strings("v__",SR,S1_2),
	append_strings(S1_2,"_",S2_2),
	append_strings(S2_2,SM1,V__M1_R),
	term_string(V2,V__M1_R),

	append_strings("v__",SR1,S1_3),
	append_strings(S1_3,"_",S2_3),
	append_strings(S2_3,SM1,V__M1_R1),
	term_string(V3,V__M1_R1),
	
	NewE=[(V1-V2),(V2-V3),(V3-V1),
	      (V2-V1),(V3-V2),(V1-V3)],
	append(E_In,NewE,E1),
	M2 is M-1,
	simplex_graph2(M2,R,E1,E_Out).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% make_symbol(+N,+B,-S)
% N = number
% B = string
% S = symbol
% transforms a number N into an atom ruleN
%
make_symbol(N,B,S):-
	number_string(N,S1),
	concat_strings(B,S1,S2),
	atom_string(S,S2).
	

