%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% computation and compilation of block graph                              % 
%                                                                         % 
% Thomas Linke                                                            % 
%                                                                         % 
% last edit: Jun 2003                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% block graph interface                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% get all grounded rules, heads and bodies (each list of IDs) 
% only correct after max_grounded!!!!!
%
gr_rule_ids( GrRuleIds ) :-
	findall(ID,( id_rule(ID,_),not(del_rule(ID)) ),GrRuleIds).

gr_head_ids( GrHeadIds ) :-
	findall(ID,( id_h(ID,_),not(del_h(ID)) ),GrHeadIds).

gr_body_ids( BodyIds ) :-
	findall(ID,( id_b(ID,_),not(del_b(ID)) ),BodyIds).

%
% deleted nodes
%
del_b( B ) :- c_del_b(B).
del_b( B ) :- nc_del_b(B).

del_h( H ) :- c_del_h(H).

del_rule( H ) :- c_del_rule(H).

%
% classification
%
facts( Fs ) :- findall(F,(c_fact(F)),Fs).

q_facts( QFs ) :- findall(F,(c_q_fact(F)),QFs).

loops_1( L1s ) :- findall(L,(c_loop_1(L)),L1s).

%
% compile_blockgraph/0 computes blockgraph with metainformation 
% from existing program database already compiled/consulted in 
% the Prolog system
%                        
compile_blockgraph :-
	!,
	max_grounded,         % maximal grounded set of nodes
	(
	    ( get_flag(reduction),get_flag(transformation) )->
	    (
		%write('graph reduction ...'),
		reduction,
		max_grounded,
		%write(' done'),nl,
		true
	    );
	    (
		true
	    )
	),
	gr_rule_ids(GrRuleIds),length(GrRuleIds,LENGTH),
%	writeln(now:LENGTH:rules:after:reduction),nl,
	write('computing block-graph ... '),
        blockgraph,           % one-to-many relations used in algorithm
	writeln('done'),nl,
	ignore_sides,         % ignore information for bodies and rules
	rule_classification.  % rule classification according to block graph

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Assumptions: LP is ordered list of rules such that a CLP rule
%              n1 : h1,...,hk <- B1;...;Bn (SYNTAX NOT CORRECT)
%              is represented as prolog term
%              rule([h1,...,hk],[B1,...,Bn],n1)
%              where B1,...,Bn are called normal bodies, 
%                    h1,...,hk are the heads of n1 and
%                    n1 (if present) is the name of the above rule 
%
% In general we have
%              rule(Head,Body,Name) 
% where Head is ordset (library ordset) of head atoms of rule r
%       Body is ordset of normal bodies prepresented by terms body(P,N) 
%       Name is the name of rule r (needed for preferences)

% Example:
%       n1 : a,b,c <- (c, not d);(x, not y)    is represented as
%       rule([a,b,c],[body([c},[d]),body([x],[y])],n1)

% Database predicates for LP and block graph:
% 
% 1. indexing:
%    id_rule/2
%    id_b/2
%    id_h/2
%
% 2. program database (meta information for block graph):
%    (one-to-one relations)
%    h_0_b/2            rule_h/2    
%    h_1_b/2            rule_b/2    
%
%    c_fact_body/1,          (c = compiled)
%    c_q_fact_body/1,     
%    c_fact_bodies/1,                
%    c_quasi_facts/1
%    loop_1/1 
%
% 3. comupte maximal grounded subset of nodes (rules,bodies,heads)
%    by deleting nodes (rules,bodies,heads) 
%    by compiling additional (ng=not grounded) ng-facts 
%    del_h/1,                   add for deleting head
%    del_b/1                    add for deleting body
%    del_rule/1,                add for deleting rule
%  
% 4. graph reduction (graph transformations):
%  
% 5. one to many relations where Xs are lists for (X = h,b,rule)
%    repecting del_X/1:
%    block graph:       meta information:
%    (redundant)        (redundant)
%    h_0_bs/2           rule_hs/2   
%    b_0_hs/2           h_rules/2   
%    h_1_bs/2           rule_bs/2   
%    b_1_hs/2           b_rules/2   
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% comupte maximal grounded subset of nodes (rules,bodies and heads)
% by compiling facts del_h/1, del_b/1 and del_rule/1 
%
max_grounded :- 
	% bodies
	get_q_facts(Qs),
	get_facts(Fs),
	ord_union(Fs,Qs,GBs1),     % facts and q-facts are bodies 
	body_ids(BodyIDs),
	ord_subtract(BodyIDs,GBs1,Bs),
	% rules
	findall(R,
	        (member(B1,GBs1),rule_b(R,B1)),
		Rs1u),
	sort(Rs1u,GRs1),
	rule_ids(RuleIDs),
	ord_subtract(RuleIDs,GRs1,Rs),
	% heads
	findall(H,
	        (
		    member(R,GRs1), 
		    rule_h(R,H)
		), 
		Hs1u),
	sort(Hs1u,GHs1),
	head_ids(HeadIDs),
	ord_subtract(HeadIDs,GHs1,Hs),

	max_grounded(Hs,GHs1,_Hsx, 
	             Bs,GBs1,  _Bsx,
		     Rs,GRs1,  _Rsx,
		     UHs,UBs,URs),

	!,
%	writeq('non-grounded heads : ':UHs),nl,
%	writeq('non_grounded bodies: ':UBs),nl,
%	writeq('non_grounded rules : ':URs),nl,

	get_del_h(UHs,DB1),
	get_del_b(UBs,DB2),
	get_del_rule(URs,DB3),
	append(DB1,DB2,DB12),
	append(DB12,DB3,DB),

	compile_clauses(DB,"tmp_del_db.pl").

get_del_h( [], [] ).
get_del_h( [UH|UHs], [c_del_h(UH)|DB1] ) :- 
	get_del_h(UHs,DB1).

get_del_b( [], [] ).
get_del_b( [UB|UBs], [c_del_b(UB)|DB1] ) :- 
	get_del_b(UBs,DB1).

get_del_rule( [], [] ).
get_del_rule( [UR|URs], [c_del_rule(UR)|DB1] ) :- 
	get_del_rule(URs,DB1).


%
% max_grounded/14 computes maximal grounded set of nodes in block graph
%
max_grounded( Hs,GHs1,GHsx, 
	      Bs,GBs1,GBsx,
	      Rs,GRs1,GRsx,
	      UHs,UBs,URs ) :-
	!,
	get_heads(GHs1,Heads),
	findall(B,
	        ( member(B,Bs),not(del_b(B)),
		  id_b(B,Body),
		  pos_body(Body,P),		  
		  ord_subset(P,Heads)),
		Bs1),
	(
	    Bs1 = [] -> % if nothing more can be grounded -> ready
	    (
		GHs1 = GHsx, 
		GBs1 = GBsx,
		GRs1 = GRsx,
		UHs = Hs, UBs = Bs, URs =Rs
	    );
	    (
		ord_subtract(Bs,Bs1,Bs2),
		ord_union(GBs1,Bs1,GBs2),
		findall(R,
	               (member(B1,Bs1),rule_b(R,B1)),
		       Rs1u),
		       sort(Rs1u,Rs1),
		ord_subtract(Rs,Rs1,Rs2),
		ord_union(GRs1,Rs1,GRs2),
		findall(H,
	               (
			   member(R,Rs1), 
			   rule_h(R,H)
		       ),
		       Hs1u),
		sort(Hs1u,Hs1),
		ord_subtract(Hs,Hs1,Hs2),
		ord_union(GHs1,Hs1,GHs2),

		max_grounded( Hs2,GHs2,GHsx, 
		              Bs2,GBs2,GBsx,
		              Rs2,GRs2,GRsx,
		              UHs,UBs,URs )
	    )
	).

%
% get_heads(+Hs,-Heads) true iff Hs ist list of head ids and 
% Heads is list of corresponding heads
%          +   -	
get_heads( [], [] ).
get_heads( [GH|GHs], [Head|Heads] ) :-
	id_h(GH,Head),
	get_heads(GHs,Heads).


%
% blockgraph:        h_0_bs/2,  b_0_hs/2,  h_1_bs/2,  b_1_hs/2
%
blockgraph :- 
	!,
	block_db(DB),
	(
	  get_flag(show) ->
	  (
	    Show = (
		     show(Node,Color,Col) :-
		         show_coloring(Node,Color,Col)
		   )
	  );
	  (
	    Show = (
		     show(_,_,_)
		   )
	  )
	),
	compile_clauses([Show|DB],"tmp_block_db.pl").

%
% block_db/1 compute one to many relations for block graph
%          -
block_db( DB ) :-
	gr_head_ids(Head_Ids),
	get_h_0_bs_and_h_1_bs(Head_Ids,DBx),
	sort(DBx,DB1),
	get_h_bs(Head_Ids,DB1ax),
	sort(DB1ax,DB1a),
	get_h_rules(Head_Ids,DB2),

	gr_body_ids(Body_Ids),
	get_b_0_hs_and_b_1_hs(Body_Ids,DBy),
	sort(DBy,DB3),
	get_b_rules(Body_Ids,DB4),

	gr_rule_ids(Rule_Ids),
	get_rule_hs_and_rule_bs(Rule_Ids,DBz), 
	sort(DBz,DB5), 

	append(DB1a,DB1,DB1b),
	append(DB1b,DB2,DBX),
	append(DB3,DB4,DBY),
	append(DBX,DBY,DBXY),
	append(DBXY,DB5,DB).

% BESSER 
mysort(X,Y) :- !,sort(X,Y).

get_h_bs( [], [] ).
get_h_bs( [ID|Head_Ids], [h_bs(ID,Bs)|DB1ax] ) :-
	findall(B,( rule_h(R,ID),not(del_rule(R)),
	            rule_b(R,B),not(del_b(B))     ),Bs1),mysort(Bs1,Bs),
	get_h_bs(Head_Ids,DB1ax).

get_h_0_bs_and_h_1_bs( [], [] ).
get_h_0_bs_and_h_1_bs( [ID|Head_Ids], [h_0_bs(ID,B0s),h_1_bs(ID,B1s)|DB] ) :-
	findall(B0,( h_0_b(ID,B0),not(del_b(B0)) ),B0s1),mysort(B0s1,B0s),
	findall(B1,( h_1_b(ID,B1),not(del_b(B1)) ),B1s1),mysort(B1s1,B1s),
	get_h_0_bs_and_h_1_bs(Head_Ids,DB).

get_b_0_hs_and_b_1_hs( [], [] ).
get_b_0_hs_and_b_1_hs( [ID|Body_Ids], [b_0_hs(ID,H0s),b_1_hs(ID,H1s)|DB] ) :-
	findall(H0,( h_0_b(H0,ID),not(del_h(H0)) ),H0s1),mysort(H0s1,H0s),
	findall(H1,( h_1_b(H1,ID),not(del_h(H1)) ),H1s1),mysort(H1s1,H1s),
	get_b_0_hs_and_b_1_hs(Body_Ids,DB).


get_rule_hs_and_rule_bs( [], [] ).
get_rule_hs_and_rule_bs( [ID|Rule_Ids], [rule_hs(ID,HS),rule_bs(ID,BS)|DB] ) :-
	findall(H,( rule_h(ID,H),not(del_h(H)) ),HS1),mysort(HS1,HS),
	findall(B,( rule_b(ID,B),not(del_b(B)) ),BS1),mysort(BS1,BS),
	get_rule_hs_and_rule_bs(Rule_Ids,DB).

get_h_rules( [], [] ).
get_h_rules( [ID|Head_Ids], [h_rules(ID,RS)|DB] ) :-
	findall(R,( rule_h(R,ID),not(del_rule(R)) ),RS1),mysort(RS1,RS),
	get_h_rules(Head_Ids,DB).

get_b_rules( [], [] ).
get_b_rules( [ID|Body_Ids], [b_rules(ID,RS)|DB] ) :-
	findall(R,( rule_b(R,ID),not(del_rule(R)) ),RS1),mysort(RS1,RS),
	get_b_rules(Body_Ids,DB).

%
% all rules, heads and bodies (IDs)
%
rule_ids( RuleIds ) :-
	findall(ID,id_rule(ID,_),RuleIds).

head_ids( HeadIds ) :-
	findall(ID,id_h(ID,_),HeadIds).

body_ids( BodyIds ) :-
	findall(ID,id_b(ID,_),BodyIds).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ignore information for sidewards propagation                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
ignore_sides :- 
	!,
	get_bs(DB1),
        get_rules(DB2),
	append(DB1,DB2,DB),
	compile_clauses(DB,"tmp_ignore_db.pl").


get_bs( DB ) :-
	gr_body_ids( GrBodyIds ),
	get_bs(GrBodyIds,DB).
get_bs( [], [] ).
get_bs( [B|Bs], [b_bs(B,SideBs)|DB] ) :-
	b_rules(B,Rs),
	findall(SB,
	        ( 
		  id_b(SB,_), SB \== B,
		  b_rules(SB,SRs),
		  ord_subset(SRs,Rs) 
	        ),
		SideBs1),
	mysort(SideBs1,SideBs),
	get_bs(Bs,DB).


get_rules( DB ) :-
	gr_rule_ids( GrRuleIds ),
	get_rules(GrRuleIds,DB).
get_rules( [], [] ).
get_rules( [R|Rs], [rule_rules(R,SideRs)|DB] ) :-
	rule_hs(R,Hs),
	findall(SR,
	        ( 
		  id_rule(SR,_), SR \== R,
		  rule_hs(SR,SHs),
		  ord_subset(SHs,Hs) 
	        ),
		SideRs1),
	mysort(SideRs1,SideRs),
	get_rules(Rs,DB).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% rule classification according to block graph                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
rule_classification :- 
	!,
	get_loop_1( DB1 ),
	get_facts(Fs),
	get_q_facts(QFs),
	append(DB1,Fs,DB2),
	append(DB2,QFs,DB3),
	compile_clauses(DB3,"tmp_class_db.pl").

%           -
get_loop_1( DB ) :-
	findall(c_loop_1(B),
	        (
		    id_b(B,_),not(del_b(B)), 
		    rule_b(R,B),not(del_rule(R)),
		    rule_h(R,H),not(del_h(H)),
		    h_1_b(H,B)
		),
		DB).

%          -
get_facts( Fs ) :-
	findall(c_fact(Id), 
	        (
		    id_b(Id,body([],_)),not(del_b(Id)),b_1_hs(Id,[]) 
		), 
		Fs).

%            -
get_q_facts( QFs ) :-
	findall(c_q_fact(Id), 
	       (
		   id_b(Id,body([],[_|_])),not(del_b(Id)),not(b_1_hs(Id,[]))
	       ), 
	       QFs).

