%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
% noMoRe parser                                                           % 
% Thomas Linke, Kathrin Konczak, Christian Anger                          %
%                                                                         %
% last edit: Jun 2002                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Syntax definition for logic programs                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Rules
rule( [], Pref, [] ) --> preference(Pref).
rule( [], [], [] ) --> compute.
rule( Rules, [], [] ) --> named_rule( Rules ).
rule( Rules, [], ClasNeg ) --> constraint_rule( Rules, ClasNeg ).

%
% Preferences
%
preference( [pref(L,R)] ) --> term([L],_),['<'], term([R],_),fullstop.

%
% compute
%
compute --> [compute],term(_,_),['{'],[not],[_false],['}'],fullstop.

%
% named_rule( Rules )
% contains an atom name(Name)
%
named_rule( Rules ) --> ic,named_body(Body,Name),
	{
	 build_named_rules(ic,Name,Body,Rules)
	}.

named_rule( Rules ) --> normal_head(Head),named_body(Body,Name),
	{
	 build_named_rules(Head,Name,Body,Rules)
	}.

%
% constraint_rule( Rules )
% may contain constraints in Head or Body
%
constraint_rule( Rules, [] ) --> ic,multbody(Body),
	{
	 build_rules(ic,Body,Rules)
	}.
constraint_rule( Rules, ClasNeg ) --> fact(Head, ClasNeg),
	{
	 build_rules(Head,[body([],[],[])],Rules)
	}.
constraint_rule( Rules, ClasNeg ) --> head(Head,ClasNeg),multbody(Body),
	{
	 build_rules(Head,Body,Rules)
	}.


%
% ic = beginning of an integerity constraint
%
ic --> [':-'].
ic --> [false],[':-'].
ic --> ['_false'],[':-'].


%
% head (may contain constraints)
%  
head( [(L,H,B)], ClasNeg ) --> nm_term((L,H,B,[]),ClasNeg),[':-'].
head( Head, ClasNeg ) --> nm_term((L,H,B,[]),CN2),[','],head(Terms, CN1),
	{
	 append([(L,H,B)],Terms,Head),
	 append(CN1,CN2,ClasNeg)
	}.

%
% normal_head (for named rules)
%  
normal_head( Head ) --> normal_term([Head]),[':-'].


%
% fact
%
fact( [(L,H,B)], ClasNeg ) --> nm_term((L,H,B,[]), ClasNeg),fullstop.
fact( Head, ClasNeg ) --> nm_term((L,H,B,[]),CN1),[','],fact(Terms,CN2),
	{
	 append([(L,H,B)],Terms,Head),
	 append(CN1,CN2,ClasNeg)
	}.


%
% nm_term for heads
%
nm_term((L,H,P,N), []) --> cterm((L,H,P,N)).
nm_term((no,no,P,[]), ClasNeg) --> term([P],ClasNeg).


%
% multbody - multiple bodies, seperated by ;
%
multbody( [Body] ) --> body(Body),
	{
	 Body=body(BP,_,_),
	 not(choose(name(_),BP,_))
	}.	 
multbody( [Body] ) --> ['('],body2(Body),
	{
	 Body=body(BP,_,_),
	 not(choose(name(_),BP,_))
	}.	 
multbody( [Body|Bodies] ) --> body(Body),multbody(Bodies),
	{
	 Body=body(BP,_,_),
	 not(choose(name(_),BP,_))
	}.	 
multbody( [Body|Bodies] ) --> ['('],body2(Body),multbody(Bodies),
	{
	 Body=body(BP,_,_),
	 not(choose(name(_),BP,_))
	}.	 


%
% named_body
%
named_body( body(NewBP,BM), Name ) --> normal_body(body(BP,BM)),
	{
	 choose(name(Name),BP,NewBP),!,
%	 not(choose(name(_Name),NewBP,_))
	 (
	   choose(name(_Name),NewBP,_) ->
	   (
	     write('parse error: multiple occurence of name/1 in rule'),nl,
	     fail
	   );true
	 )
	}.

%
% normal_body
%
normal_body( body(PBody, []) )    --> normal_term(PBody),fullstop.
normal_body( body([], NBody) )    --> not(_),normal_term(NBody),fullstop.
normal_body( body(PBody, NBody) ) --> normal_term(T),[','],normal_body(body(P,NBody)),
	{
            append(T,P,PBody)
	}.
normal_body( body(PBody, NBody) ) --> not(_),normal_term(T),[','],
	                              normal_body(body(PBody,N)),
	{
            append(T,N,NBody)
	}.


%
% body (may contain constraints)
%
body( body([], [], [Card]) ) --> cterm(Card),end_of_body. %cardinality or weigth constraint
body( body(PB, NB, Card2) ) --> cterm(Card),[','],body(body(PB,NB,Card1)),
	{
            append(Card1,[Card],Card2)
	}.
body( body(PBody, [], []) ) --> term(PBody,_),end_of_body.
body( body([], NBody, []) ) --> not(_),term(NBody,_),end_of_body.
body( body(PBody, NBody, Card) ) --> term(T,_),[','],body(body(P,NBody,Card)),
	{
            append(T,P,PBody)
	}.
body( body(PBody, NBody, Card) ) --> not(_),term(T,_),[','],body(body(PBody,N,Card)),
	{
            append(T,N,NBody)
	}.

body2( body([], [], [Card]) ) --> cterm(Card),end_of_body2. %cardinality or weigth constraint
body2( body(PB, NB, Card2) ) --> cterm(Card),[','],body2(body(PB,NB,Card1)),
	{
            append(Card1,[Card],Card2)
	}.
body2( body(PBody, [], []) ) --> term(PBody,_),end_of_body2.
body2( body([], NBody, []) ) --> not(_),term(NBody,_),end_of_body2.
body2( body(PBody, NBody, Card) ) --> term(T,_),[','],body2(body(P,NBody,Card)),
	{
            append(T,P,PBody)
	}.
body2( body(PBody, NBody, Card) ) --> not(_),term(T,_),[','],body2(body(PBody,N,Card)),
	{
            append(T,N,NBody)
	}.


%
% end_of_body
%
end_of_body --> fullstop.
end_of_body --> [';'].
end_of_body2 --> [')'],fullstop.
end_of_body2 --> [')'],[';'].


%
% cterm
%
cterm( (0,0,PB,NB) ) --> ['{'],cbody( PB, NB ).
cterm( (0,0,PB,NB) ) --> ['['],wbody( PB, NB ).
cterm( (Low,0,PB,NB) ) --> integer(Low),['{'],cbody( PB, NB ).
cterm( (Low,0,PB,NB) ) --> integer(Low),['['],wbody( PB, NB ).
cterm( (0,Up1,PB,NB) ) --> ['{'],cbody( PB, NB ),integer(Up),
	           {
	             Up1 is Up + 1
	           }.
cterm( (0,Up1,PB,NB) ) --> ['['],wbody( PB, NB ),integer(Up),
	           {
	             Up1 is Up + 1
	           }.
cterm( (Low,Up1,PB,NB) ) --> integer(Low),['{'],cbody( PB, NB ),integer(Up),
	           {
	             Up1 is Up + 1
	           }.
cterm( (Low,Up1,PB,NB) ) --> integer(Low),['['],wbody( PB, NB ),integer(Up),
	           {
	             Up1 is Up + 1
	           }.

%
% cbody
%
cbody( [(PBody,1)], [] ) --> term([PBody],_),['}'].
cbody( [], [(NBody,1)] ) --> not(_),term([NBody],_),['}'].
cbody( PBody, NBody ) --> term([T],_),[','],cbody(P,NBody),
	{
            append([(T,1)],P,PBody)
	}.
cbody( PBody, NBody ) --> not(_),term([T],_),[','],cbody(PBody,N),
	{
            append([(T,1)],N,NBody)
	}.

%
% wbody
%
wbody( PBody, [] ) --> wterm(PBody),[']'].
wbody( [], NBody ) --> not(_),wterm(NBody),[']'].
wbody( PBody, NBody ) --> wterm(T),[','],wbody(P,NBody),
	{
            append(T,P,PBody)
	}.
wbody( PBody, NBody ) --> not(_),wterm([T]),[','],wbody(PBody,N),
	{
            append(T,N,NBody)
	}.

%
% wterm
%
wterm( [(Term,Weight)] ) --> term([Term],_CN),['='],integer(Weight).

%
% normal_term
%
normal_term( Term )   --> atom( Term ).
normal_term( Term )   --> atom( T ),['('],list_of_terms(Terms),
	{
            T = [Functor],
	    T1 =.. [Functor|Terms],
	    Term = [T1]
        }.
normal_term( Term )   --> ['('],normal_term( Term ),[')'].

%
% term (can be classicaly negated)
%
term( Term, ClasNeg )   --> cn_atom( Term, ClasNeg ).
term( Term, [] )   --> atom( Term ).
term( Term, ClasNeg )   --> cn_atom( T, ClasNeg ),['('],list_of_terms(Terms),
	{
            T = [Functor],
	    T1 =.. [Functor|Terms],
	    Term = [T1]
        }.
term( Term, [] )   --> atom( T ),['('],list_of_terms(Terms),
	{
            T = [Functor],
	    T1 =.. [Functor|Terms],
	    Term = [T1]
        }.
term( Term, ClasNeg )   --> ['('],term( Term, ClasNeg ),[')'].

%
% list_of_terms
%
list_of_terms( Term )  --> atom(Term),[')'].
list_of_terms( Term )  --> inner_term(Term),[')'].
list_of_terms( Terms ) --> atom(Atom),[','],list_of_terms(List),
	{
            append(Atom,List,Terms)
	}. 
list_of_terms( Terms ) --> inner_term(T),[','],list_of_terms(List),
	{
            append(T,List,Terms)
	}.

inner_term( Term )   --> atom( T ),['('],list_of_terms(Terms),
	{
            T = [Functor],
	    T1 =.. [Functor|Terms],
	    Term = [T1]
        }.


%
% cn_atom (atom, that may contain classical negation)
%
cn_atom( [CN_Atom], ClasNeg) --> [-],[Atom],
	{
	 atom(Atom),
	 not(Atom == not),
	 not(Atom == false),
	 classical_negation(Atom,ClasNeg,CN_Atom)
	}. 

	 

%
% atom
%
atom( [Atom] ) --> [Atom],
	{
            (atom(Atom);integer(Atom)),
	    not(Atom == not),
	    not(Atom == false)
	}. % not is no atom
atom( [Atom] ) --> sign(S), integer(I),
	{
 	   (
 	       (S == -) -> 
 	       (
 		   Atom is 0 - I % negative
 	       );
 	       (
 		   Atom is I
 	       )
 	   )
        }.

sign( S ) --> [S],
        {
            member(S,[+,-])
	}.

%
% not
%
not( [not] )   --> [not].

%
% integer
% attention: we have tow different integer/1 an eclipse builtin and
% a parser integer/1. DO NOT CONFUSE!!!!!!! 
% 
integer( I ) --> [I],
	{
            integer(I)
        }.

%
% fullstop
%
fullstop       --> [fullstop].

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% procedures needed while parsing                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
% build_rule(+Name,+Head,+BP,+BM,-Rule)
%
build_rule(Name,Head,BP,BM,Rule):-
	rule_name(Rule,Name),
	rule_head(Rule,[Head]),
	list_to_ord_set(BP,SBP),
	list_to_ord_set(BM,SBM),
	pos_body(Body,SBP),
	neg_body(Body,SBM),
	rule_body(Rule,[Body]).

%
% build_rule(+Name,+Heads,+Bodies,-Rule)
%
build_rule(Name,Heads,Bodies,Rule):-
	rule_name(Rule,Name),
	list_to_ord_set(Heads,SHeads),
	rule_head(Rule,SHeads),
	rule_body(Rule,Bodies).
%
%  build_named_rules(+Head,+Name,+Body,-Rules)
%
%
build_named_rules(ic,Name,body(BP,BM),[Rule]):-
	new_atom(IC),
	build_rule(Name,IC,BP,[IC|BM],Rule).
build_named_rules(Head,Name,body(BP,BM),[Rule]):-
	build_rule(Name,Head,BP,BM,Rule).


%
% build_rules(+Heads,+Bodies,-Rules)
% expands constraints in bodies and heads
%
build_rules(Heads, Bodies, Rules) :-
	handle_bodies(Bodies,NewBodies,Rules1),
	handle_heads(Heads,NewBodies,Rules2),
	append(Rules1,Rules2,Rules).


%
% handle_bodies(+Bodies,-NewBodies,-Contraint_Rules)
% checks each body for constraints and handles them
%
handle_bodies([],[],[]).
handle_bodies([Body|Bodies],NewBodies,Rules):-
	Body=body(BP,BM,Constraints),
	handle_constraints(Constraints,Add_BP,Add_BM,Rules1),
	append(BP,Add_BP,New_BP),
	append(BM,Add_BM,New_BM),
	intersection(New_BP,New_BM,Intersection),
	(
	  Intersection = [] -> % not an nonapplicable rule
                               % like x :- a, not a ?
	  (
	    list_to_ord_set(New_BP,SBP),
	    list_to_ord_set(New_BM,SBM),
	    pos_body(NewBody,SBP),
	    neg_body(NewBody,SBM)
	  );
	  (
	    pos_body(NewBody,Intersection),
	    neg_body(NewBody,Intersection)
	  )
	),
	NewBodies=[NewBody|Out],
	handle_bodies(Bodies,Out,Rules2),
	append(Rules1,Rules2,Rules).



%
% handle_constraints(+Constraints, -AddBP, -AddBM, -AddRules)
% finds all combinations of atom in a Constraint that sattisfy the
% bounderies, builds rules for those and gives new parts of the
% body of the original rule
%
handle_constraints([],[],[],[]).
handle_constraints([(LowerBound,UpperBound,CPBIn,CNBIn)|More],PB,NB,Rules):-
	sort(2,>=,CPBIn,CPB),
	sort(2,>=,CNBIn,CNB),
	addall(CPB,Length1),
	addall(CNB,Length2),
	Length is Length1 + Length2,
	(
	    LowerBound=0 ->
	    (
		(UpperBound=0;UpperBound>Length) ->
		(
		    % no bounds given
		    PB=PB_Out,NB=NB_Out,Rules=Rules_Out
		);
		(
		    % only upper bound given
		    findall(Combi,
			    get_combinations(CPB,CNB,UpperBound,Combi),
			    UpCombinations),
		    new_constraint_atom(Int2),
		    combine(Int2,UpCombinations,Int2Clauses),
		    append(Int2Clauses,Rules_Out,Rules),
		    PB=PB_Out,NB=[Int2|NB_Out]
		)
	    );
	    (
		(UpperBound=0;UpperBound>Length) ->
		(
		    % only lower bound
		    findall(Combi,
			    get_combinations(CPB,CNB,LowerBound,Combi),
			    LowCombinations),
		    new_constraint_atom(Int1),
		    combine(Int1,LowCombinations,Int1Clauses),
		    append(Int1Clauses,Rules_Out,Rules),
		    NB=NB_Out,PB=[Int1|PB_Out]
		);
		(
		    %both bounds 
		    findall(Combi,
			    get_combinations(CPB,CNB,LowerBound,Combi),
			    LowCombinations),
		    findall(Combi,
			    get_combinations(CPB,CNB,UpperBound,Combi),
			    UpCombinations),
		    new_constraint_atom(Int1),
		    new_constraint_atom(Int2),
		    combine(Int1,LowCombinations,Int1Clauses),
		    combine(Int2,UpCombinations,Int2Clauses),
		    append(Int1Clauses,Int2Clauses,Rules1),
		    append(Rules1,Rules_Out,Rules),
		    PB=[Int1|PB_Out],NB=[Int2|NB_Out]
		)
	    )
	),
	handle_constraints(More,PB_Out,NB_Out,Rules_Out).


%
% handle_heads( +Heads, +Bodies, -Rules)
% checks Heads for Constraints
%
handle_heads(ic,Bodies,[Rule]):-
	new_atom(IC),
	noname(Name),
	add_to_each_nb(Bodies,IC,NewBodies),
	build_rule(Name,[IC],NewBodies,Rule).
handle_heads(Heads,Bodies,Rules):-
	handle_heads2(Heads,Bodies,NewHeads,Rules1),
	(
	  NewHeads = [] ->
	  (
	    Rules=Rules1
	  );
	  (
	    noname(Name),
	    build_rule(Name,NewHeads,Bodies,Rule),
	    Rules=[Rule|Rules1]
	  )
	).
	
handle_heads2([],_Bodies,[],[]):-!.
handle_heads2([(no,no,Head)|Heads],Bodies,[Head|NewHeads],Rules):-
	handle_heads2(Heads,Bodies,NewHeads,Rules).
handle_heads2([Head|Heads],Bodies,NewHeads,Rules):-
	handle_head(Head,Bodies,Rules1),
	handle_heads2(Heads,Bodies,NewHeads,Rules2),
	append(Rules1,Rules2,Rules).


%
% handle_head( +Head, +Bodies, -Rules)
%
handle_head((Low,High,Atoms),Bodies,Rules):-
	addall(Atoms,H),
	(
	    High=0 ->
	    (
		H1 is H + 1
	    );
	    (
		H1=High
	    )
	),
	for_each_atom_get_in_or_out(Atoms,Bodies,Rules1),
	% ic falls mehr als obere Schranke drin
	build_rules(ic,[body([],[],[(H1,0,Atoms,[])])],Rules2),
	% ic falls abs-low+1 raus
	Low1 is H - Low + 1,
	build_rules(ic,[body([],[],[(Low1,0,[],Atoms)])],Rules3),
	append(Rules1,Rules2,Rules4),
	append(Rules4,Rules3,Rules).


for_each_atom_get_in_or_out([],_Bodies,[]).
for_each_atom_get_in_or_out([(A,_)|As],Bodies,[Rule1,Rule2|IoO]):-
	(
	    atom(A) ->
	    (
		concat_atoms(constraintnot_,A,New)
	    );
	    (
		A=..[Atom|L],
		concat_atoms(constraintnot_,Atom,NewAtom),
		New=..[NewAtom|L]
	    )
	),
	noname(Name),
	add_to_each_nb(Bodies,New,Bodies1),
	build_rule(Name,[A],Bodies1,Rule1),
	noname(Name1),
%	add_to_each_nb(Bodies,A,Bodies2),
	build_rule(Name1,New,[],[A],Rule2),
	for_each_atom_get_in_or_out(As,Bodies,IoO).

%
% add_to_each_nb(+Bodies,+Add,-NewBodies)
%
add_to_each_nb([],_Add,[]).
add_to_each_nb([Body|Bodies],Add,[NewBody|NewBodies]):-
	neg_body(Body,Neg),
	pos_body(Body,Pos),
	pos_body(NewBody,Pos),
	ord_insert(Neg,Add,NewNeg),
	neg_body(NewBody,NewNeg),
	add_to_each_nb(Bodies,Add,NewBodies).

	

%
% addall/2 adds all weights in the constraint
%
addall([],0).
addall([(_,W)|Rest],Z):-
	addall(Rest,Z1),
	Z is W + Z1.

%
% get_combinations/4 gets all combinations of constraint_atoms, so that the
% accumulated weight is at least Z
%
get_combinations(_,_,Z,([],[])):-(Z =< 0),!.
get_combinations([],[],_,([],[])):-fail.
get_combinations(PB,NB,I,(AOut1,AOut2)):-!,
	my_choose(P,M,(PB,NB),(Rest1,Rest2)),
	mysubtract(I,P,M,I1),
	(
	    P=[] ->
	    (
		M=[(E,_W)],
		AOut2=[E|Out2],
		AOut1=Out1
	    );
	    (
		P=[(E,_W)],
		AOut1=[E|Out1],
		AOut2=Out2
	    )
	),
	get_combinations(Rest1,Rest2,I1,(Out1,Out2)).

%
% my_choose/4 chooses an constraint_atom
%
my_choose([I],[],([I|L],P),(L,P)).
my_choose(I1,I2,([_|L],P),(R,P1)):-
	my_choose(I1,I2,(L,P),(R,P1)).
my_choose([],[I],([],[I|P]),([],P)).
my_choose(I1,I2,([],[_|P]),([],R1)):-
	my_choose(I1,I2,([],P),([],R1)).

%
% combine/3 combines heads and body atoms
%
combine(_Head,[],[]).
combine(Head,[(L1,L2)|Ls],[Rule|Rules]):-
	noname(Name),
	build_rule(Name,Head,L1,L2,Rule),
	combine(Head,Ls,Rules).

%
% my_subtract/4 needed for get_combinations
%
mysubtract(I,[],[],I).
mysubtract(I,[(_P,W1)],[],I1):-!,
	I1 is I - W1.
mysubtract(I,[],[(_P,W2)],I1):-!,
	I1 is I - W2.

%
% classical_negation/3 needed for occurances of classical negation
%
classical_negation(Atom,ClasNeg,CN_Atom):-
	concat_atoms('classicaly_negated_',Atom,CN_Atom),
	ClasNeg = [cn(CN_Atom,Atom)].


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% parse_file(+File,-LP) parses File and returns logic program LP 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
parse_file(File,Theory,CN):-
	parse_file(File,rule,Theory,CN).

%
% parse_file(+File,+DCG,-Theory) performs syntax check with DCG on File 
% and returns Theory   
%
parse_file( File, DCG, Theory, ClasNeg ):-
	read_tokens(File,TokenSets),
	my_phrase(DCG,TokenSets,Theory1,Prefs1,ClasNeg),
	sort(Theory1,Theory),     % for dealing with multiple rule occurence
	sort(Prefs1,Prefs),        % for dealing with multiple rule occurence
	(
	  true ->
	  (
	    length(Theory,Length1),
	    length(Prefs,Length2),
	    write(Length1),
	    (
	      Length1>1 ->
	      (
		write(' rules and ')
	      );
	      (
		write(' rule and ')
	      )
	    ),
	    write(Length2),
	    (
	      Length2>1 ->
	      (
		writeln(' preferences successfully parsed ')
	      );
	      (
		writeln(' preference successfully parsed ')
	      )
	    )
	  );true
	),
	!.

%
% my_phrase(+DCG,+TokenSets,-Theory,-Prefs,-ClasNeg)
%
my_phrase( _,[],[],[],[] ) :- !.
my_phrase( DCG, [TSet|RestTSets], Rules, Prefs, ClasNeg ):-
	DCG_local=..[DCG,Rules1,Prefs1,ClasNeg1], 
	phrase(DCG_local,TSet),
	!,
	my_phrase(DCG,RestTSets,Rules2,Prefs2,ClasNeg2),
	append(Rules1,Rules2,Rules),
	append(Prefs1,Prefs2,Prefs),
	append(ClasNeg1,ClasNeg2,ClasNeg).
my_phrase( _DCG, [TSet|_RestTSets], _Rules, _Prefs, _ClasNeg ):-
	nl,write('Syntax Error in clause: '),nl,
        write(TSet),nl,
	!,
	fail.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% read_tokens/2 read tokens TokenSets from File
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
read_tokens( File,TokenSets ):- 
		open(File,read,Stream),
		read_file_token(32,_,Stream,TokenSets),
		close(Stream).

read_file_token(CharIn,CharOut, Stream, TokenSets ) :-     
	read_token_set(CharIn,Ch,Stream,TS),
	!,
	(
	    not(TS = [end_of_file]) ->
	    (
		TokenSets = [TS|RestSets],
		read_file_token(Ch,CharOut,Stream,RestSets)
	    );
	    (
		TokenSets = [],
                CharOut=Ch
	    )
	).

read_file_token(_,_, _Stream, [] ) :- % sollte nicht vorkommen
	write('ERROR: read_file_token failed'),nl,nl,
	!,fail.

read_token_set(CharIn,CharOut, Stream,[NewT|Rest] ):-
	my_read_token(CharIn,Ch,Stream,T,Class),
%	read_token(Stream,T,Class),
	(
	    T = end_of_file ->
	    (
		NewT = T,Rest = []
	    );
	    (
		not(Class=fullstop) ->
		(
		    process_token(T,NewT),
		    !,
		    read_token_set(Ch,CharOut,Stream,Rest)
		);
		(
		    NewT=fullstop,
		    Rest=[],
		    CharOut=Ch
		)
	    )
	).

process_token( Token, Token ):- atom(Token),!.
process_token( Token, Token ):- integer(Token),!.
process_token( Token, NewToken ):-
	string(Token),!,
	atom_string(NewToken,Token).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% tokenizer for swi-prolog and sicstus
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% my_read_token( +Char_In, -Char_Out, +Stream, -Token, -Class ) 
%
% reads a token, that begins with the character Char_In from the
% stream Stream and gives back the token and its class 
% The character Char_Out, that is the first character after the
% token Token, is needed for reading the next token recurrent
% through read_token_set/4
%

my_read_token(ChrIn,ChrOut,Stream, Token, Class ) :-
	tokenize(ChrIn,ChrOut,Stream,Token,Class).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% tokenize(+Actual_Char,+Character_after_token,+Stream,-Token,-Class)
%

tokenize(-1,-1, _, end_of_file, end_of_file):-!.          % end_of_file

tokenize(Char,ChrOut,Stream,Tokens,Class):-               % ignore
	Char =< 31,
	get0(Stream,NextCh),
	tokenize(NextCh,ChrOut,Stream,Tokens,Class).

tokenize(32,ChrOut,Stream,Tokens,Class):-                 % space
	get0(Stream,Char),
	tokenize(Char,ChrOut,Stream,Tokens,Class).

tokenize(37,ChrOut,Stream,Tokens,Class) :-!, 		  %  %comment
	repeat,					          %  
	    get0(Stream,Ch),		                  %  stop when 
	    ( Ch = 10 ; Ch = -1 ),		          %    end_of_line ( Char = 10)
	!,					          %  or
	( Ch =\= -1                                       %end_of_file(Char=-1)
          ->
	  (  
	     get0(Stream,NextCh),                         %    but then fail
	     tokenize(NextCh,ChrOut,Stream,Tokens,Class)  % 

	  );
	  (
	     tokenize(-1,ChrOut,Stream,Tokens,Class)
	  )
        ).


tokenize(47,ChrOut,Stream,Tokens,Class):-                 % /* ... */ comment
	get0(Stream,NCh),NCh == 42,                       
	get0(Stream,Next),
	read_comment(Next,Stream,Chr),		          
	tokenize(Chr,ChrOut,Stream,Tokens,Class).


tokenize(46,ChrOut,Stream,fullstop,fullstop):-            % fullstop 
	get0(Stream,ChrOut).                              % end of a token set
                                                          % for logic programs

tokenize(58,ChrOut,Stream,':-',token):-                   % :-
	get0(Stream,NCh),NCh == 45,                       % for logic programs
	get0(Stream,ChrOut).

tokenize(59,ChrOut,Stream,';',token):-                    % ;
                                                          % for nested
	                                                  % logic programs
	get0(Stream,ChrOut).

tokenize(Char,ChrOut,Stream,Str,token):-                  % beginning of a string, where   
	(   Char=95;                                      %    _
	    (Char >= 65, Char =< 90);                     %    A..Z
	    (Char >= 97, Char =< 122);                    %    a..z
	    (Char >= 48, Char =< 57)),                    %    0..9
	read_token_string(Char,ChrOut,Stream,Name),               % are only contained
	name(Str,Name).

tokenize(Char,ChrOut,Stream,Ch,token):-                   % the remaining characters
	name(Ch,[Char]),                                  % are unchanged given back
	get0(Stream,ChrOut).
	


%
% reading a string ( only the ASCII- Code of it)
% and give back the character after this string
%
read_token_string(Ch,ChrOut,Stream,[Ch|Name]):-                   
	(   Ch=95;                              % _
	    (Ch >= 65, Ch =< 90);               % A..Z
	    (Ch >= 97, Ch =< 122);              % a..z
	    (Ch >= 48, Ch =< 57)                % 0..9
	),!,             
	get0(Stream,NCh),
	read_token_string(NCh,ChrOut,Stream,Name).
read_token_string(Ch,Ch,_,[]).



%
% reading the /* ... */ comment
% and give back the character after this comment
%
read_comment(42,Stream,ChrOut):-
	get0(Stream,NCh),NCh == 47,
	get0(Stream,ChrOut).

read_comment(_Char,Stream,ChrOut):-
	get0(Stream,NCh),
	read_comment(NCh,Stream,ChrOut).

%
% new_atom/1 returns some new atom
%
new_atom( A ) :-
	!,
	inc_counter(newatom),
	new_atom_name(NA),
	get_counter(newatom,V),
	make_symbol(V,NA,A).

new_constraint_atom( A ) :-
	!,
	inc_counter(newconstraintatom),
	new_constraint_atom_name(NA),
	get_counter(newconstraintatom,V),
	make_symbol(V,NA,A).

make_symbol(Number,String,Symbol):-
	number_string(Number,String1),
	concat_strings(String,String1,String2),
	atom_string(Symbol,String2).

noname(noname).
new_atom_name("nomore_atom").
new_constraint_atom_name("constraint_atom").

