/*  Copyright (c) June 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifndef BIN_TREE_H
#define BIN_TREE_H

#include <cassert>
#include <new>
#include <memory>
#include <iterator>
#include <algorithm>

namespace toolbox
{
	// You need to specialize this struct
	// for your own traversal type
	template<class Tag>
	struct bin_tree_traversal
	{
		template<class U>
		static U* forward(U*) { return 0; }
		template<class U>
		static U* backward(U*) { return 0; }
	};

	namespace detail
	{
		template<class T>
		struct bin_tree_node
		{
			bin_tree_node(	const T& t = T(), 
							bin_tree_node* parent = 0,
							bin_tree_node* left = 0,
							bin_tree_node* right = 0)
							:	parent_(parent)
							,	left_(left)
							,	right_(right)
							,	t_(t)
			{}
			
			bin_tree_node* parent_;
			bin_tree_node* left_;
			bin_tree_node* right_;
			T t_;
		};

		template<class T, class Reference = T&, class Pointer = T*, class NodeType = bin_tree_node<T> >
		class bin_tree_cursor
		{
		public:
			typedef T value_type;
			typedef Reference reference;
			typedef Pointer pointer;
			typedef NodeType node_type;
			bin_tree_cursor(node_type* node = 0)
				:	current_(node)
			{}
			
			inline reference operator*()
			{
				return current_->t_;
			}
			inline pointer operator->()
			{
				return &current_->t_;
			}
			inline bool good() const { return current_ != 0; }
			inline bool leaf() const 
			{ 
				return current_->left_ == 0 && current_->right_ == 0; 
			}
			inline bool has_parent() const { return current_->parent_ != 0; }
			inline bool has_left_child() const { return current_->left_ != 0; }
			inline bool has_right_child() const { return current_->right_ != 0; }
			inline bin_tree_cursor& descend_left()
			{
				current_ = current_->left_;
				return *this;
			}
			inline bin_tree_cursor& descend_right()
			{
				current_ = current_->right_;
				return *this;
			}
			inline bin_tree_cursor& ascend()
			{
				current_ = current_->parent_;
				return *this;
			}
			inline bin_tree_cursor left() const
			{
				return bin_tree_cursor(current_->left_);
			}
			inline bin_tree_cursor right() const
			{
				return bin_tree_cursor(current_->right_);
			}
			inline bin_tree_cursor parent() const
			{
				return bin_tree_cursor(current_->parent_);
			}
			inline bin_tree_cursor& operator=(node_type* pos)
			{
				current_ = pos;
				return *this;
			}
			inline void swap(bin_tree_cursor& other)
			{
				std::swap(current_, other.current_);
			}
			inline node_type* node() { return current_; }
			inline const node_type* node() const { return current_; }
		protected:
			inline void node(node_type* n) { current_ = n; } 
		private:
			node_type* current_;
		};

		template<class T, class U, class V, class W>
		inline bool operator==(const bin_tree_cursor<T,U,V,W>& lhs, const bin_tree_cursor<T,U,V,W>& rhs)
		{
			return lhs.node() == rhs.node();
		}

		template<class T, class U, class V, class W>
		inline bool operator!=(const bin_tree_cursor<T,U,V,W>& lhs, const bin_tree_cursor<T,U,V,W>& rhs)
		{
			return lhs.node() != rhs.node();
		}

		template<	class TraversalTraits, 
					class T, class Reference = T&, class Pointer = T*, class NodeType = bin_tree_node<T> >
		class bin_tree_iterator : public bin_tree_cursor<T, Reference, Pointer, NodeType>
		{
		public:
			typedef bin_tree_cursor<T, Reference, Pointer, NodeType> base_type;
			typedef typename base_type::node_type node_type;
			typedef std::bidirectional_iterator_tag iterator_category;
			typedef long difference_type;

			inline bin_tree_iterator& operator++()
			{
				base_type::node(TraversalTraits::forward(base_type::node()));
				return *this;
			}
			bin_tree_iterator operator++(int)
			{
				bin_tree_iterator temp(*this);
				++(*this);
				return temp;
			}
			inline bin_tree_iterator& operator--()
			{
				base_type::node(TraversalTraits::backward(base_type::node()));
				return *this;
			}
			bin_tree_iterator operator--(int)
			{
				bin_tree_iterator temp(*this);
				--(*this);
				return temp;
			}
		protected:
			bin_tree_iterator(node_type* node = 0)
				:	base_type(node)
			{}
		};


		template<class T, class TraversalTag>
		struct bin_tree_iterator_traversal_traits_base
		{
			typedef bin_tree_node<T> node_type;
			typedef bin_tree_traversal<TraversalTag> traversal_type;
		};

		template<class T, class TraversalTag>
		struct bin_tree_iterator_forward_traversal_traits : public bin_tree_iterator_traversal_traits_base<T, TraversalTag>
		{
			typedef bin_tree_iterator_traversal_traits_base<T, TraversalTag> base_type;
			typedef typename base_type::node_type node_type;
			typedef typename base_type::traversal_type traversal_type;
			inline static node_type* forward(node_type* node)
			{
				return traversal_type::forward(node);
			}
			inline static node_type* backward(node_type* node)
			{
				return traversal_type::backward(node);
			}
		};

		template<class T, class TraversalTag>
		struct bin_tree_iterator_backward_traversal_traits : public bin_tree_iterator_traversal_traits_base<T, TraversalTag>
		{
			typedef bin_tree_iterator_traversal_traits_base<T, TraversalTag> base_type;
			typedef typename base_type::node_type node_type;
			typedef typename base_type::traversal_type traversal_type;
			inline static node_type* forward(node_type* node)
			{
				return traversal_type::backward(node);
			}
			inline static node_type* backward(node_type* node)
			{
				return traversal_type::forward(node);
			}
		};
		
		template<class T>
		struct bin_tree_iterator_type_traits
		{
			typedef T value_type;
			typedef T& reference;
			typedef T* pointer;
			typedef bin_tree_node<T> node_type;
		};

		template<class T>
		struct bin_tree_const_iterator_type_traits
		{
			typedef T value_type;
			typedef const T& reference;
			typedef const T* pointer;
			typedef bin_tree_node<T> node_type;
		};

		template<class T, class TraversalTag>
		struct bin_tree_forward_iterator_traits		:	public bin_tree_iterator_type_traits<T>
													,	public bin_tree_iterator_forward_traversal_traits<T, TraversalTag>
		{
			typedef bin_tree_iterator_type_traits<T> type_traits;
			typedef typename type_traits::value_type value_type;
			typedef typename type_traits::reference reference;
			typedef typename type_traits::pointer pointer;
			typedef typename type_traits::node_type node_type;
			typedef bin_tree_iterator_forward_traversal_traits<T, TraversalTag> traversal_traits;
			typedef bin_tree_iterator<
				traversal_traits,
				value_type,
				reference,
				pointer,
				node_type> iterator;
		};

		template<class T, class TraversalTag>
		struct bin_tree_backward_iterator_traits	:	public bin_tree_iterator_type_traits<T>
													,	public bin_tree_iterator_backward_traversal_traits<T, TraversalTag>
		{
			typedef bin_tree_iterator_type_traits<T> type_traits;
			typedef typename type_traits::value_type value_type;
			typedef typename type_traits::reference reference;
			typedef typename type_traits::pointer pointer;
			typedef typename type_traits::node_type node_type;
			typedef bin_tree_iterator_backward_traversal_traits<T, TraversalTag> traversal_traits;
			typedef bin_tree_iterator<
				traversal_traits,
				value_type,
				reference,
				pointer,
				node_type> iterator;
		};


		template<class T, class TraversalTag>
		struct bin_tree_forward_const_iterator_traits	:	public bin_tree_const_iterator_type_traits<T>
														,	public bin_tree_iterator_forward_traversal_traits<T, TraversalTag>
		{
			typedef bin_tree_const_iterator_type_traits<T> type_traits;
			typedef typename type_traits::value_type value_type;
			typedef typename type_traits::reference reference;
			typedef typename type_traits::pointer pointer;
			typedef typename type_traits::node_type node_type;
			typedef bin_tree_iterator_forward_traversal_traits<T, TraversalTag> traversal_traits;
			typedef bin_tree_iterator<
				traversal_traits,
				value_type,
				reference,
				pointer,
				node_type> iterator;
		};

		template<class T, class TraversalTag>
		struct bin_tree_backward_const_iterator_traits	:	public bin_tree_const_iterator_type_traits<T>
														,	public bin_tree_iterator_backward_traversal_traits<T, TraversalTag>
		{
			typedef bin_tree_const_iterator_type_traits<T> type_traits;
			typedef typename type_traits::value_type value_type;
			typedef typename type_traits::reference reference;
			typedef typename type_traits::pointer pointer;
			typedef typename type_traits::node_type node_type;
			typedef bin_tree_iterator_backward_traversal_traits<T, TraversalTag> traversal_traits;
			typedef bin_tree_iterator<
				traversal_traits,
				value_type,
				reference,
				pointer,
				node_type> iterator;
		};
	}

	
	struct bin_tree_inorder_tag {};
	struct bin_tree_preorder_tag {};
	struct bin_tree_leafs_only_tag {};

	template<>
	struct bin_tree_traversal<bin_tree_leafs_only_tag>
	{
		template<class U>
		static U* forward(U* current_)
		{
			if(!current_)
				return 0;

			// move as far down the tree as possible
			if(current_->left_)
				return forward(current_->left_);

			if(current_->right_)
				return forward(current_->right_);
			
			// only way to go is up
			// move from down right to left up and find the next right turn
			current_ = find_next_right(current_);

			// the right turn was valid
			if(current_)
			{
				// find the beginning from this subtree on
				if(current_->left_)
					return begin(current_->left_);
				if(current_->right_)
					return begin(current_->right_);

				return current_;
			}
			return 0;
		}
		template<class U>
		static U* backward(U* current_)
		{
			if(!current_)
				return 0;
			
			if(current_->right_)
				return backward(current_->right_);

			if(current_->left_)
				return backward(current_->left_);
			
			current_ = find_next_left(current_);
			if(current_)
			{
				if(current_->right_)
					return rbegin(current_->right_);
				if(current_->left_)
					return rbegin(current_->left_);
				return current_;
			}
			return 0;
		}
		template<class U>
		static U* begin(U* root)
		{
			if(!root)
				return 0;

			if(root->left_)
				return begin(root->left_);

			if(root->right_)
				return begin(root->right_);

			return root;
		}
		template<class U>
		inline static U* end(U* root)
		{
			return 0;
		}
		template<class U>
		static U* rbegin(U* root)
		{
			if(!root)
				return 0;

			if(root->right_)
				return rbegin(root->right_);

			if(root->left_)
				return rbegin(root->left_);
			
			return root;
		}
		template<class U>
		inline static U* rend(U* root)
		{
			return 0;
		}
	private: 
		template<class U>
		static U* find_next_right(U* current_)
		{
			if(!current_)
				return 0;

			// Abbruchbedingung
			if(current_->right_)
				return current_->right_;

			// gehe von rechts unten nach links oben im Baum so weit wie mglich
			while(current_->parent_ && current_->parent_->right_ == current_)
				current_ = current_->parent_;

			// gehe einen weiter (hoch links unten nach rechts oben)
			// und suche erneut
			return find_next_right(current_->parent_);
		}
		template<class U>
		static U* find_next_left(U* current_)
		{
			if(!current_)
				return 0;

			if(current_->left_)
				return current_->left_;

			while(current_->parent_ && current_->parent_->left_ == current_)
				current_ = current_->parent_;

			return find_next_left(current_->parent_);
		}
	};

	template<>
	struct bin_tree_traversal<bin_tree_preorder_tag>
	{
		template<class U>
		static U* forward(U* current_)
		{
			if(!current_)
				return 0;

			if(current_->left_)
				return current_->left_;					
			
			return find_next_right(current_);			
		}
		template<class U>
		static U* backward(U* current_)
		{
			if(!current_)
				return 0;

			if(current_->parent_ && current_->parent_->right_ == current_ && current_->parent_->left_)
				return rbegin(current_->parent_->left_);
			
			return current_->parent_;
		}
		template<class U>
		inline static U* begin(U* root)
		{
			return root;
		}
		template<class U>
		inline static U* end(U* root)
		{
			return 0;
		}
		template<class U>
		static U* rbegin(U* root)
		{
			if(!root)
				return 0;

			if(root->right_)
				return rbegin(root->right_);
			else if(root->left_)
				return rbegin(root->left_);
			return root;
		}
		template<class U>
		inline static U* rend(U* root)
		{
			return 0;
		}
	private: 
		template<class U>
		static U* find_next_right(U* current_)
		{
			if(!current_)
				return 0;

			// Abbruchbedingung
			if(current_->right_)
				return current_->right_;

			// gehe von rechts unten nach links oben im Baum so weit wie mglich
			while(current_->parent_ && current_->parent_->right_ == current_)
				current_ = current_->parent_;

			// gehe einen weiter (hoch links unten nach rechts oben)
			// und suche erneut
			return find_next_right(current_->parent_);
		}
	};

	template<>
	struct bin_tree_traversal<bin_tree_inorder_tag>
	{
		template<class U>
		static U* forward(U* current_)
		{
			if(!current_)
				return 0;
			
			if(current_->right_)
				return begin(current_->right_);
			
			while(current_->parent_ && current_->parent_->right_ == current_)
				current_ = current_->parent_;
			
			return current_->parent_;
		}
		template<class U>
		static U* backward(U* current_)
		{
			if(!current_)
				return 0;
			
			if(current_->left_)
				return rbegin(current_->left_);
			
			while(current_->parent_ && current_->parent_->left_ == current_)
				current_ = current_->parent_;
			
			return current_->parent_;
		}
		template<class U>
		inline static U* begin(U* root)
		{
			while(root && root->left_)
				root = root->left_;
			return root;
		}
		template<class U>
		inline static U* end(U* root)
		{
			return 0;
		}
		template<class U>
		inline static U* rbegin(U* root)
		{
			while(root && root->right_)
				root = root->right_;
			return root;
		}
		template<class U>
		inline static U* rend(U* root)
		{
			return 0;
		}
	};
	
	


	template<class T, class TraversalTag>
	class bin_tree_forward_iterator : public detail::bin_tree_forward_iterator_traits<T, TraversalTag>::iterator
	{
	public:
		typedef typename detail::bin_tree_forward_iterator_traits<T, TraversalTag>::iterator base_type;
		typedef typename base_type::node_type node_type;
		bin_tree_forward_iterator(node_type* node = 0)
			:	base_type(node)
		{}
		template<class U>
		bin_tree_forward_iterator(const U& other)
			:	base_type(const_cast<node_type*>(other.node()))
		{}
		template<class U>
		inline bin_tree_forward_iterator& operator=(const U& other)
		{
			node(const_cast<node_type*>(other.node()));
			return *this;
		}
	};
	template<class T, class TraversalTag>
	class bin_tree_backward_iterator : public detail::bin_tree_backward_iterator_traits<T, TraversalTag>::iterator
	{
	public:
		typedef typename detail::bin_tree_backward_iterator_traits<T, TraversalTag>::iterator base_type;
		typedef typename base_type::node_type node_type;
		bin_tree_backward_iterator(node_type* node = 0)
			:	base_type(node)
		{}
		template<class U>
		bin_tree_backward_iterator(const U& other)
			:	base_type(const_cast<node_type*>(other.node()))
		{}
		template<class U>
		inline bin_tree_backward_iterator& operator=(const U& other)
		{
			node(const_cast<node_type*>(other.node()));
			return *this;
		}
	};

	template<class T, class TraversalTag>
	class bin_tree_forward_const_iterator : public detail::bin_tree_forward_const_iterator_traits<T, TraversalTag>::iterator
	{
	public:
		typedef typename detail::bin_tree_forward_const_iterator_traits<T, TraversalTag>::iterator base_type;
		typedef typename base_type::node_type node_type;
		bin_tree_forward_const_iterator(node_type* node = 0)
			:	base_type(node)
		{}
		template<class U>
		bin_tree_forward_const_iterator(const U& other)
			:	base_type(const_cast<node_type*>(other.node()))
		{}
		template<class U>
		inline bin_tree_forward_const_iterator& operator=(const U& other)
		{
			node(const_cast<node_type*>(other.node()));
			return *this;
		}
	};

	template<class T, class TraversalTag>
	class bin_tree_backward_const_iterator : public detail::bin_tree_backward_const_iterator_traits<T, TraversalTag>::iterator
	{
	public:
		typedef typename detail::bin_tree_backward_const_iterator_traits<T, TraversalTag>::iterator base_type;
		typedef typename base_type::node_type node_type;
		bin_tree_backward_const_iterator(node_type* node = 0)
			:	base_type(node)
		{}
		template<class U>
		bin_tree_backward_const_iterator(const U& other)
			:	base_type(const_cast<node_type*>(other.node()))
		{}
		template<class U>
		inline bin_tree_backward_const_iterator& operator=(const U& other)
		{
			node(const_cast<node_type*>(other.node()));
			return *this;
		}
	};


	template<class T>
	inline bool operator==(const detail::bin_tree_cursor<T>& lhs, const detail::bin_tree_cursor<T>& rhs)
	{
		return lhs.node() == rhs.node();
	}
	template<class T>
	inline bool operator!=(const detail::bin_tree_cursor<T>& lhs, const detail::bin_tree_cursor<T>& rhs)
	{
		return !(lhs == rhs);
	}

	template<class T, class A = std::allocator<T> >
	class bin_tree
	{
	public:
		typedef detail::bin_tree_node<T> node_type;
		
		typedef detail::bin_tree_cursor<T> cursor;
		typedef detail::bin_tree_cursor<T, const T&, const T*> const_cursor;
		typedef T value_type;
		typedef T& reference;
		typedef const T& const_reference;
		typedef size_t size_type;
		typedef typename A::template rebind<node_type> rebind_type;
		typedef typename rebind_type::other allocator_type;
		
		~bin_tree()
		{
			erase(root_);
		}
		bin_tree()
			:	root_(0)
		{}
		bin_tree(const T& t)
			:	root_(create(t, 0))
		{}
		bin_tree(const bin_tree& other)
			:	root_(copy(other.root_, 0))
		{}
		template<class Cursor>
		bin_tree(Cursor pos)
			:	root_(copy(pos.node(), 0))
		{}
		
		inline cursor root() { return cursor(root_); }
		inline const_cursor root() const { return const_cursor(root_); }
		inline cursor nil() { return cursor(0); }
		inline const_cursor nil() const { return const_cursor(0); }

		cursor root(const_reference t)
		{
			if(root_)
			{	
				root_->t_ = t;
				erase(root_->left_);
				erase(root_->right_);
			}
			else
			{
				root_ = create(t, 0);
			}
			return cursor(root_);
		}
		
		template<class U>
		inline bin_tree_forward_iterator<T, U> begin() 
		{ 
			return bin_tree_forward_iterator<T, U>(bin_tree_traversal<U>::begin(root_)); 
		}
		template<class U>
		inline bin_tree_forward_iterator<T, U> end() 
		{ 
			return bin_tree_forward_iterator<T, U>(bin_tree_traversal<U>::end(root_)); 
		}
		template<class U>
		inline bin_tree_forward_const_iterator<T, U> begin() const 
		{ 
			return bin_tree_forward_const_iterator<T, U>(bin_tree_traversal<U>::begin(root_)); 
		}
		template<class U>
		inline bin_tree_forward_const_iterator<T, U> end() const 
		{ 
			return bin_tree_forward_const_iterator<T, U>(bin_tree_traversal<U>::end(root_)); 
		}
	

		template<class U>
		inline bin_tree_backward_iterator<T, U> rbegin() 
		{ 
			return bin_tree_backward_iterator<T, U>(bin_tree_traversal<U>::rbegin(root_)); 
		}
		template<class U>
		inline bin_tree_backward_iterator<T, U> rend() 
		{ 
			return bin_tree_backward_iterator<T, U>(bin_tree_traversal<U>::rend(root_)); 
		}
		template<class U>
		inline bin_tree_backward_const_iterator<T, U> rbegin() const 
		{ 
			return bin_tree_backward_const_iterator<T, U>(bin_tree_traversal<U>::rbegin(root_)); 
		}
		template<class U>
		inline bin_tree_backward_const_iterator<T, U> rend() const 
		{ 
			return bin_tree_backward_const_iterator<T, U>(bin_tree_traversal<U>::rend(root_)); 
		}
	
	
		inline void clear()
		{
			// no way to guard against a throwing d'tor		
			erase(root_);
		}
		inline bool empty() const { return root_ == 0; }
		inline size_type size() const
		{
			return size(root_);
		}
	
		template<class Iterator>
		inline Iterator insert_left(Iterator pos, const_reference t)
		{
			return Iterator(insert_left(pos.node(), t));
		}
		template<class Iterator>
		inline Iterator insert_right(Iterator pos, const_reference t)
		{
			return Iterator(insert_right(pos.node(), t));
		}
		template<class Iterator1, class Iterator2>
		inline Iterator1 insert_left(Iterator1 pos, Iterator2 subtree)
		{
			return Iterator1(insert_left(pos.node(), subtree.node()));
		}
		template<class Iterator1, class Iterator2>
		inline Iterator1 insert_right(Iterator1 pos, Iterator2 subtree)
		{
			return Iterator1(insert_right(pos.node(), subtree.node()));
		}

		template<class Iterator1, class Iterator2>
		Iterator1 splice_left(Iterator1 pos, bin_tree& other, Iterator2 subtree)
		{
			node_type* where = pos.node();
			if(where->left_)
				return Iterator1();

			node_type* node = subtree.node();
			splice_common(other, node);
			where->left_ = node;
			node->parent_ = where;
			return Iterator1(node);
		}
		template<class Iterator1, class Iterator2>
		Iterator1 splice_right(Iterator1 pos, bin_tree& other, Iterator2 subtree)
		{
			node_type* where = pos.node();
			if(where->right_)
				return Iterator1();

			node_type* node = subtree.node();
			splice_common(other, node);
			where->right_ = node;
			node->parent_ = where;
			return Iterator1(node);
		}
		
		inline void erase(cursor pos)
		{
			erase(pos.node());
			if(pos.node() == root_)
				root_ = 0;
		}
		inline void swap(bin_tree& other)
		{
			std::swap(root_, other.root_);
			std::swap(allocator_, other.allocator_);
		}
		bin_tree& operator=(const bin_tree& other)
		{
			if(this == &other)
				return *this;
			bin_tree(other).swap(*this);
			return *this;
		}
		inline size_type max_size() const
		{
			return allocator_.max_size();
		}
	private:
		static size_type size(node_type* root)
		{
			if(!root)
				return 0;
			
			return 1 + size(root->left_) + size(root->right_);
		}
		node_type* insert_left(node_type* where, const_reference t)
		{
			if(where->left_)
				return 0;
			where->left_ = create(t, where);
			return where->left_;
		}
		node_type* insert_right(node_type* where, const_reference t)
		{
			if(where->right_)
				return 0;
			where->right_ = create(t, where);
			return where->right_;
		}
		node_type* insert_left(node_type* where, node_type* root)
		{
			if(where->left_)
				return 0;
			where->left_ = copy(root, where);
			return where->left_;
		}
		node_type* insert_right(node_type* where, node_type* root)
		{
			if(where->right_)
				return 0;
			where->right_ = copy(root, where);
			return where->right_;
		}
		static void splice_common(bin_tree& other, node_type* node)
		{
			if(node == other.root_)
			{
				other.root_ = 0;
			}
			else
			{
				if(node->parent_->left_ == node)
					node->parent_->left_ = 0;
				else
					node->parent_->right_ = 0;
			}
		}
		void erase(node_type* where)
		{
			if(where)
			{
				erase(where->left_);
				erase(where->right_);

				if(where->parent_)
				{
					if(where->parent_->left_ == where)
						where->parent_->left_ = 0;
					else
						where->parent_->right_ = 0;
				}

				destroy(where);
			}
		}
		node_type* copy(node_type* node, node_type* parent = 0)
		{
			if(!node)
				return 0;
			
			node_type* new_node = create(node->t_, parent);
			try
			{
				new_node->left_ = copy(node->left_, new_node);
				new_node->right_ = copy(node->right_, new_node);
			}
			catch(...)
			{
				erase(new_node);
				throw;
			}
			return new_node;
		}
		inline node_type* create(const_reference t, node_type* parent = 0)
		{
			return new (allocator_.allocate(1, static_cast<void*>(0))) node_type(t, parent);
		}
		inline void destroy(node_type* node)
		{
			node->~node_type();
			allocator_.deallocate(node, 1);
		}
	private:
		allocator_type allocator_;
		node_type* root_;
	};

	namespace detail
	{
		template<class T>
		bool equal(const bin_tree_node<T>* lhs, const bin_tree_node<T>* rhs)
		{
			if(	(lhs == 0 && rhs != 0) ||
				(lhs != 0 && rhs == 0))
			return false;

			if(!lhs)
				return true;

			assert(lhs);
			assert(rhs);

			return	lhs->t_ == rhs->t_ && equal(lhs->left_, rhs->left_) && equal(lhs->right_, rhs->right_);
		}
	}

	template<class T, class A>
	inline bool operator==(const bin_tree<T, A>& lhs, const bin_tree<T, A>& rhs)
	{
		return detail::equal(lhs.root().node(), rhs.root().node());
	}

	template<class T, class A>
	inline bool operator!=(const bin_tree<T, A>& lhs, const bin_tree<T, A>& rhs)
	{
		return !(lhs == rhs);;
	}

}

#endif
