/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_OPERATOR_H
#define NOMORE_OPERATOR_H

#include <string>
#include <set>

namespace NS_NOMORE {
class Graph;

//! The base class for all nomore operators
class Operator {
public:
  //! Creates a new operator.
  /*! \param name The name of the concrete operator type */
  explicit Operator(const std::string& name);
  
  //! Creates a new operator.
  /*! \param name The name of the concrete operator type 
   *  \param graph The Graph the new operator should work on.
   */
  Operator(const std::string& name, Graph& graph);
  
  //! Destroys the operator.
  virtual ~Operator();

  //! Executes this operator.
  /*! \return true if the execution of this operator changed the color of 
   *  one or more nodes. Otherwise false. 
   */
  bool operator()();
  
  typedef std::set<std::string> OpSet;
  //! returns true if this operator is compatible with all operators in ops
  /*!
   * \param ops a set of operator names.
   * \return the default implementation returns true, meaning that this operator
   * does not depend on any operator (except from forward propagation).
   * \note override this method to enforce operator-dependencies.
   */
  virtual bool validate(const OpSet& ops) const;
  
  //! returns the name of the concrete operator type
  const std::string& getName() const {
    return name_;
  }

  //! return this if name is equal to this->getName()
  virtual Operator* extractOperator(const std::string& name);

protected:
  void setName(const std::string& name) {
    name_ = name;
  }
private:
  Operator(const Operator&);
  Operator& operator=(const Operator&);
  
  struct ExecutionGuard;
  friend struct ExecutionGuard;
  struct ExecutionGuard {
    ExecutionGuard(Operator* self)
      : self_(self) {
    }
    ~ExecutionGuard() {
      self_->postExecute();
    }
    Operator* self_;
  };

  //! called after the execution of this operator.
  /*!
   * overwrite this method if you need to restore your operator to a specific
   * state once its execution finished.
   */
  virtual void postExecute();
  
  //! overwrite this method to implement the behaviour of your concrete operator
  virtual bool execute() = 0;
  std::string name_;

  
};

//! convenience base class for all operators needing a reference to the body-head graph.
class GraphOperator : public Operator {
public:
  //! Creates a new operator.
  /*! \param name The name of the concrete operator type 
   *  \param graph The Graph the new operator should work on.
   */
  GraphOperator(const std::string& name, Graph& grp);

  //! returns the graph this operator works on
  const Graph& getGraph() const {
    return *graph_;
  }
  
  //! returns the graph this operator works on
  Graph& getGraph() {
    return *graph_;
  }
private:
  //! The graph to work on.
  Graph *graph_;
};

}

#endif
