/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_STACK_H
#define LIBNOMORE_STACK_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <color.h>

#include <stack>
#include <vector>
#include <utility>
#include <cassert>
namespace NS_NOMORE {

class Node;
class Graph;

struct BacktrackingMode {
  //! Mode values.
  enum Mode {
    //! the strategie of smodels
    symmetric,
    //! the strategie of dlv
    asymmetric
  };
};

//! A backtracking stack is a stack of branches storing backtrack-information for nodes.
/*! 
 *  A backtracking stack (BacktrackStack) consists of a stack 
 *	of \ref BacktrackStack::Branch "branches".
 *	\ref BacktrackStack::Branch "Branches" correspond to choice points. Whenever a
 *	a new choice point is encountered a new branch is pushed
 *	onto the backtrack stack so that each decision level is represented
 *	by one \ref BacktrackStack::Branch "branch".
 *	A	\ref BacktrackStack::Branch "branch" on the other hand contains information
 *	about colored nodes. Whenever a node is colored an \ref BacktrackStack::Item "item"
 *  is pushed on the top branch.
 *	That is while exploring the consequences of the coloring of a choice point
 *	you keep pushing \ref BacktrackStack::Item "items" on the active \ref BacktrackStack::Branch "branch stack".
 * 
 *  Backtracking from the last choice point means restoring all items (and therefore nodes)
 *	on the top \ref BacktrackStack::Branch "branch stack" and then popping the
 *	branch. After the top branch was popped off the backtack stack, the choice
 *	point is colored to its alternative color which yields a new item (but no choice point)
 *	that is pushed onto the now active branch stack.
 */
class BacktrackStack {
public:
  //! Simple item of a \ref BacktrackStack::Branch "backtrack branch".
  /*! An Item stores the information needed to restore the color of one node. */
  struct Item {
  		Item(Node& n, Color::ColorValue oldColor)
  			: node_(&n)
  			, color_(oldColor) {
      }
  		Node *node_;
      Color::ColorValue color_;
	};
  
  //! Creates an empty backtrack stack.
  BacktrackStack();

  //! Returns true if this backtrack stack is empty, i.e. does not contain any branches.
  bool isEmpty() const {
    return branches_.empty();
  }
  
  //! Pushes the item i to the top branch if such a branch exists.
  void pushItem(const Item& i) {
    if (!branches_.empty()) 
      branches_.top().first.push(i);
  }

  //! Pushes a new branch to this backtrack stack.
  /*! \param i the Item for which the new branch is created (a choice point).
   *  \param altColor the alternative color for the new branch. */
  void pushBranch(const Item& i, Color::ColorValue altColor) {
    branches_.push(BranchStack::value_type(Branch(i), altColor));
	}

  //! Restores all nodes contained in the top branch.
  /*! \pre A branch exists, i.e. isEmpty() != false.
   *  \note The cp that startet the top branch is not restored.
   *  \see Branch::restore */
  void restore(Graph& g)  {
		assert(!branches_.empty());
		branches_.top().first.restore(g);
	}

  //! Restores all nodes contained in the top branch.
  /*! Restores the cp that started the top branch as well as all nodes contained 
   * in that branch. Then pops that branch from the backtrack stack.
   * \pre A branch exists, i.e. isEmpty() != false.
   * \return An Item containing the alternative color of the restored branch and
   * the cp node that started the branch, i.e. the cp and its alternative color. */
  Item restoreAndPop(Graph& g) {
    assert(!branches_.empty());
    Color::ColorValue alt = branches_.top().second;
    Node* cp = branches_.top().first.restoreCp(g);
    branches_.pop();
  	return Item(*cp, alt);
  }
  
private:
  BacktrackStack(const BacktrackStack&);
	BacktrackStack& operator=(const BacktrackStack&);
	
	//! A branch of the backtracking stack holding a list of nodes.
  /*! A branch holds all information necessary to restore the consequences of
   *  the coloring of one choice point as well as the choice point itself. */
  class Branch {
	public:
		//! Creates a new branch for the Item cp.
    explicit Branch(const Item& cp);
		
		//! Pushes the Item n to this branch.
    void push(const Item& n) {
			stack_.push(n);
		}
		
    //! Restores all nodes in this branch.
    /*! \note The node that started this branch is not restored. */
    void restore(Graph& g);

    //! Restores the Item that started this branch and all nodes of the branch.
    /*! \return Returns the node that started this branch. */
    Node* restoreCp(Graph& g);

	private:
    std::stack<Item, std::vector<Item> > stack_;
    Item cp_;
	};
	typedef std::pair<Branch, Color::ColorValue> BranchItem;
  typedef std::stack<BranchItem, std::vector<BranchItem> > BranchStack;
  BranchStack branches_;
};

} // NS_NOMORE

#endif
