/*	$Id: algorithm.h 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef PLATYPUS_ALGORITHM_H
#define PLATYPUS_ALGORITHM_H

#include <memory>
#include <iostream>
#include <string>
#include <fstream>
#include <cassert>
#include <program_options/program_options.h>
#include <program_options/value.h>
#include <platypus/nop_stream.h>
#include <platypus/time.h>
#include <platypus/platypus_algorithm_options.h>
#include <platypus/platypus_types_factory.h>
#include <platypus/platypus_algorithm_callback.h>
#include <platypus/types/exceptions.h>
#include <platypus/types/chooser.h> // for CHOOSER_POLICY_EXPANDER

#define PRIVATE_PLATYPUS_ALGORITHM
#include <platypus/private/platypus_algorithm.h>
#undef PRIVATE_PLATYPUS_ALGORITHM

namespace Platypus
{
	const char* const NAME = "PLATYPUS";

	template<class DistributionTraits>
	class PlatypusAlgorithm
	{
	private:
		PlatypusAlgorithm(int argc, char** argv)
			:	cmdlOptions_(createOptions())
			,	cmdlValues_(new ProgramOptions::OptionValues)
			,	str_(&std::cout)
		{
			// allow the distribution traits and the core to specify command line options
			// it is interested in.
			DistributionTraits::addCommandlineOptions(*cmdlOptions_);
			parseCommandline(argc, argv);
		
			if(!options_.wantHelpOnOptions())
			{
				// enable debug printing if wanted
				if(options_.debugPrinting())
					str_.enable();

				std::istream* input = &std::cin;
				std::ifstream file;
				if(!options_.filename().empty())
				{
					file.open(options_.filename().c_str());
					input = &file;
					assert(file);
				}
				
				ProgramInterface& program = PlatypusTypesFactory::instance().programFactory().create(*input, options_.expander());
				
				// to have concrete type
				DistributionTraits* dt = new DistributionTraits(
					argc, argv, *cmdlValues_, options_.requestedAnswerSets(), str_);
				traits_.reset(dt);
				traits_->program(program);
				
				core_.reset(PlatypusTypesFactory::instance().coreFactory().create(
					*dt, 
					program, 
					PlatypusTypesFactory::instance(), 
					options_, 
					str_,
					options_.core()));

				if(options_.silent())
				{
					core_->disableAnswerSetPrinting();
				}
			}
		}
	public:
		static PlatypusAlgorithm* instance(int argc, char** argv)
		{
			instance_.reset(new PlatypusAlgorithm(argc, argv));
			setSignalHandler(&signalHandler);
			return instance_.get();
		}
		static PlatypusAlgorithm* instance()
		{
			return instance_.get();
		}
		void run()
		{
			if(!options_.wantHelpOnOptions())
			{
				time_.start();
				traits_->setup();
				core_->run();
				traits_->teardown();
				time_.stop();
			}
		}
		void shutdown()
		{
			if(!options_.wantHelpOnOptions())
				traits_->terminate();
		}
	private:
		size_t expanderInitializations() const { return traits_->expanderInitializations(); }
		size_t conflicts() const { return traits_->conflicts(); }
		size_t answerSetsFound() const { return traits_->answerSetsFound(); }
		size_t backtracks() const { return traits_->backtracks(); }
		size_t threads() const { return traits_->threads(); }
		size_t threadDelegations() const { return traits_->threadDelegations(); }
		double duration() const { return ( time_.difference()*1.0)/(time_.frequency()*1.0); }
		const std::string& localChoicePolicy() const 
		{ 
			static const std::string expander(CHOOSER_POLICY_EXPANDER);
			if(ExpanderTypeInfo::canChoose(options_.expander()))
				return expander;
			return options_.localChoicePolicy(); 
		}
		const std::string& dcChoicePolicy() const { return options_.delegatableChoicePolicy(); }
		unsigned delegationThreshold() const { return options_.percentage(); }
		bool usesBacktracking() const { return ExpanderTypeInfo::canBacktrack(options_.expander()); }
		bool suppressAnswerSetPrinting() const { return options_.silent(); }
		const std::string& core() const { return core_->type(); }
	public:
		void setOutputStream(std::ostream& os)
		{
			str_.str(os);
			if(!options_.wantHelpOnOptions())
				core_->setOutputStream(os);
		}
		void enableDebugPrinting() 
		{
			str_.enable();
		}
		void disableDebugPrinting() 
		{ 
			str_.disable();
		}
		static std::string usage() { return usageString(); }	
		std::ostream& print(std::ostream& os) const;
	private:
		static void signalHandler(int)
		{
			ignoreFurtherSignals();
			if(PlatypusAlgorithm::instance())
			{
				PlatypusAlgorithm::instance()->shutdown();
			}
		}
		PlatypusAlgorithm();
		void parseCommandline(int argc, char** argv)
		{
			try
			{
				cmdlValues_->store(ProgramOptions::parseCommandLine(argc, argv, *cmdlOptions_, true));
			}
			catch(const exception& e)
			{
				throw CommandlineError(e.what());
			}
			checkProgramOptions(*cmdlValues_);
			options_.setOptions(*cmdlValues_);
		}
	private:
		std::auto_ptr<PlatypusAlgorithmDistributionCallback> traits_;
		std::auto_ptr<PlatypusAlgorithmCoreCallback> core_;
		std::auto_ptr<ProgramOptions::OptionGroup> cmdlOptions_;
		std::auto_ptr<ProgramOptions::OptionValues> cmdlValues_;
		PlatypusAlgorithmOptions options_;
		Time time_;
		NopStream str_;

		static std::auto_ptr<PlatypusAlgorithm> instance_;		
	};

	template<class T>
	std::auto_ptr< PlatypusAlgorithm<T> > PlatypusAlgorithm<T>::instance_;

	template<class T>
	std::ostream& PlatypusAlgorithm<T>::print(std::ostream& os) const
	{
		if(!options_.wantHelpOnOptions())
		{
			if(traits_->print()) // If we want this instance to print...
			{
				os << "\nThis is PLATYPUS flavor >>" << traits_->type() << "<<.\n";
				os << "Run configuration:\n";
				os << "\tCore: " << core() << "\n";
				os << "\tBacktracking: " << (usesBacktracking() ? "yes" : "no") << "\n";
				os << "\tLocal choice policy: " << localChoicePolicy() << "\n";
				os << "\tDelegatable choice policy: " << dcChoicePolicy() << "\n";
				os << "\tDelegation threshold: " << delegationThreshold() << "\n";
				os << "\tPrint answer sets: " << (!suppressAnswerSetPrinting() ? "yes" : "no") << "\n";
				os << "\n";
				os << "Run statistics:\n";
				os << "\tAnswer sets found: " << answerSetsFound() << "\n";
				os << "\tExpander initializations: " << expanderInitializations() << "\n";
				os << "\tBacktracks: " << backtracks() << "\n";
				os << "\tConflicts: " << conflicts() << "\n";
				os << "\tTotal time: " << duration() << "\n";
				os << "\n";
				os << "Core statistics:\n";
				os << "\tNumber of threads: " << threads() << "\n";
				os << "\tDelegations to other threads: " << threadDelegations() << "\n";
				os << "\n";
				os << "Distribution statistics:\n";
				os << *traits_ << "\n";
			}
		}
		else
		{
			os << *cmdlOptions_ << "\n";
		}
		return os;
	}

	template<class T>
	std::ostream& operator<<(std::ostream& os, const PlatypusAlgorithm<T>& platypus)
	{
		return platypus.print(os);
	}
}

#endif
