/*	$Id: algorithm.cpp 2241 2005-07-20 10:00:33Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <platypus/algorithm/algorithm.h>
#include <iostream>
#include <fstream>
#include <cassert>
#include <nop_stream.h>
#include <platypus/exceptions.h>
#include <platypus/options.h>
#include <platypus/distribution_base.h>
#include <platypus/core_base.h>
#include <platypus/program_base.h>
#include <platypus/algorithm/cout_printer.h>
#include <platypus/algorithm/pretty_print.h>
#include <platypus/algorithm/version.h>


#ifdef _MSC_VER
#	pragma warning(disable:4267) // size_t -> unsigned conversion
#endif

using namespace std;

namespace Platypus
{
	PlatypusAlgorithm::~PlatypusAlgorithm()
	{
		// quiet incomplete type warnings
	}
	PlatypusAlgorithm::PlatypusAlgorithm(	std::auto_ptr<NopStream> str, 
											std::auto_ptr<ProgramInterface> program, 
											std::auto_ptr<PlatypusOptions> options, 
											std::auto_ptr<CoreBase> core,
											std::auto_ptr<DistributionBase> distribution)
											:	str_(str)
											,	program_(program)
											,	options_(options)
											,	core_(core)
											,	distribution_(distribution)
											,	printer_(new CoutPrinter)
	{
		if(!str_.get() || !program_.get() || !options_.get() || !core_.get() || !distribution_.get())
			throw NullPointerException();

		printer_->stream(str_->str());
		distribution_->printer(*printer_);
		openInput();
	}
	void PlatypusAlgorithm::openInput()
	{
		if(!options_->filename().empty())
		{
			ifstream input(options_->filename().c_str());
			if(!input)
				throw InputError(string("Could not open file '") + options_->filename() + "' for reading!");
			program_->setup(input);
		}
		else
		{
			program_->setup(cin);
		}
	}

	void PlatypusAlgorithm::setOutputStream(std::ostream& os)
	{
		str_->str(os);
		printer_->stream(os);
	}
	void PlatypusAlgorithm::enableDebugPrinting()
	{
		str_->enable();
	}
	void PlatypusAlgorithm::disableDebugPrinting() 
	{ 
		str_->disable();
	}
	void PlatypusAlgorithm::run()
	{
		// prepare
		totalTime_.start();
		distribution_->setup();
		core_->setup();

		// run
		calculationTime_.start();
		core_->run();
		calculationTime_.stop();

		// clean up
		distribution_->teardown();
		totalTime_.stop();
	}
	void PlatypusAlgorithm::shutdown()
	{
		distribution_->terminate();
	}

	std::ostream& PlatypusAlgorithm::print(std::ostream& os) const
	{
		if(distribution_->print()) // If we want this instance to print...
		{
			os	<< "\nThis is PLATYPUS version " 
				<< PLATYPUS_VERSION_MAJOR << "." << PLATYPUS_VERSION_MINOR << "." << PLATYPUS_VERSION_PATCHLEVEL << ".\n";
			os << "Run configuration:\n";
			os << "\tDistribution: " << distribution_->type() << "\n";
			os << "\tCore: " << core_->type() << "\n";
			os << "\tExpander: " << options_->expander() << "\n";
			os << "\tBacktracking: " << (PrettyPrint::expanderCanBacktrack(options_->expander()) ? "yes" : "no") << "\n";
			os << "\tLocal choice policy: " << options_->localChoicePolicy() << "\n";
			os << "\tDelegatable choice policy: " << options_->delegatableChoicePolicy() << "\n";
			os << "\tDelegation threshold: " << options_->percentage() << "\n";
			os << "\tPrint answer sets: " << (options_->silent() ? "no" : "yes") << "\n";
			os << "\tSearch space probing: ";
			if(options_->probing())
			{
				os	<< "yes\n";
				os	<< "\tProbing parameters:\n";
				os	<< "\t\tinitial cutoff = " << options_->probingInitialCutoff() << ",\n";
				os  << "\t\tsteps = " << options_->probingSteps() << ",\n";
				os	<< "\t\tbest n = ";
				if(options_->probingBestN())
					os << options_->probingBestN() << ",\n";
				else
					os << "off,\n";

				os	<< "\t\tpolicy = " << options_->probingPolicy() << "\n";
			}
			else
			{
				os << "no\n";
			}
			os << "\n";
			os << "Run statistics:\n";
			os << "\tAtoms: " << program_->numberOfAtoms() << "\n";
			os << "\tRules: " << program_->numberOfRules() << "\n";
			os << "\tAnswer sets found: " << distribution_->answerSetsFound() << "\n";
			os << "\tExpander initializations: " << distribution_->expanderInitializations() << "\n";
			os << "\tBacktracks: " << distribution_->backtracks() << "\n";
			os << "\tConflicts: " << distribution_->conflicts() << "\n";
			os << "\tTotal time: " << ((totalTime_.difference()*1.0)/(totalTime_.frequency()*1.0)) << "\n";
			os << "\tCalculation time: " << ((calculationTime_.difference()*1.0)/(calculationTime_.frequency()*1.0)) << "\n";
			os << "\n";
			os << "Core statistics:\n";
			os << "\tNumber of threads: " << (PrettyPrint::multipleThreads(options_->core()) ? options_->threads() : 1) << "\n";
			os << "\tDelegations to other threads: " << distribution_->threadDelegations() << "\n";
			os << "\n";
			os << "Distribution statistics:\n";
				distribution_->print(os);
				os << "\n\n";
		}
		return os;
	}
}
