/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_THREAD_H
#define PT_THREAD_H
#include <memory>
#include <stdexcept>
#include <portablethreads/config.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/thread.h>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/thread.h>
#endif
#include <portablethreads/utility.h>
#include <portablethreads/mutex.h>

PT_NAMESPACE_BEGIN
const int PT_THREAD_OK = 0;
const int PT_THREAD_ALREADY_RUNNING = -1;
const int PT_THREAD_NOT_RUNNING = -2;
const int PT_THREAD_DEADLOCK = -3;
const int PT_THREAD_RESOURCE = -4;
const int PT_THREAD_ALREADY_CREATED = -5;

const int PT_THREAD_RUNNING = 1;
const int PT_THREAD_STOPPED = 2;
const int PT_THREAD_DESTROYED = 3;

class PThread
{	
	friend class PTPrivate::ThreadManager;

	typedef PTPrivate::OSThreadTraits OSThreadTraits;
public:
	typedef PTPrivate::thread_id_t thread_id_t;
	typedef OSThreadTraits::os_id_t os_id_t;
	static inline os_id_t self()
	{
		return OSThreadTraits::self();
	}
	inline thread_id_t id() const
	{
		return id_;
	}
public:
	PT_DLL_EXPORT virtual ~PThread();
	PT_DLL_EXPORT PThread();
    
	// Start the thread.
	// This method returns IMMEDIATELY, it does NOT wait until the thread actually runs.
	// At any point in time there may be only one active thread per PThread object.
	PT_DLL_EXPORT int run();
	
	// Terminate (brute force) the thread. If the thread doesn't run this is a noop.
	// If a thread is stopped using this method it is stopped dead in its tracks. 
	// This means that among other things no d'tors for local objects are called.
	// Use with care!
	PT_DLL_EXPORT void stop();

	// Wait for a thread to finish. 
	PT_DLL_EXPORT int join();

	// thread information
	PT_DLL_EXPORT static int poll(thread_id_t id);
protected:
	// Give up the time slice that is left to a thread.
	inline void give()
	{
		OSThreadTraits::yield();
	}
	inline bool stopped() const
	{
		return poll(id_) == PT_THREAD_STOPPED;
	}
	inline bool running() const 
	{
		return poll(id_) == PT_THREAD_RUNNING;
	}
private:
	// You need to subclass PThread an implement this method.
	// This will get called by the thread
	PT_DLL_EXPORT virtual void threadImpl() = 0;
	PT_DLL_EXPORT virtual void unexpectedException() throw();
private:
	// Entry point for all threads
	PT_DLL_EXPORT static OSThreadTraits::entry_function_t entry;
	// common static code for all threads
	PT_DLL_EXPORT static void entry_common(void* arg);
	inline void id(thread_id_t id)
	{
		id_ = id;
	}
	inline void setObjectDestructionNotifier(volatile bool* notifiy)
	{
		notifyOnDestruction_ = notifiy;
	}
	 
	// Threads cannot be copied or assigned ;-)
	PThread(const PThread&);
	PThread& operator=(const PThread&);		
private:
	volatile thread_id_t id_;
	volatile bool* notifyOnDestruction_;
private:
	PT_DLL_EXPORT static std::auto_ptr<PTPrivate::ThreadManager>  manager_;
};	


// Comparison operators for threads
inline bool operator==(const PThread& lhs, const PThread& rhs)
{
	return lhs.id() == rhs.id();
}

inline bool operator!=(const PThread& lhs, const PThread& rhs)
{
	return !(lhs == rhs);
}

// This exception gets thrown by PThreadObjectSemantic
// if a thread could not be created. This shouldn't ever
// happen but it is better to notify the user if it does.
class CouldNotCreateThread : public std::runtime_error
{
public:
	CouldNotCreateThread(const std::string& what = "[PThreadObjectSemantic] A thread could NOT be created")
		:	std::runtime_error(what)
	{}
};

// Policy class for PThreadObjectSemantic
// There are currently three different policies:
// 1.	RunOnce -> invokes loopImpl() once
// 2.	RunN -> invokes loopImpl() N times
// 3.	RunForever -> repeatedly invoked loopImpl() 
//		util the thread gets terminated
// 
// If you want to provide your own policies, create
// a class that has three public methods:
// 1.	void reset(); -> resets the policy to initial state
// 2.	bool next(); -> is evaluated in each iteration. This
//		method should return true for as long as you want the
//		thread to run
// 3.	void shutdown(); -> must make next() return false
template<unsigned int N>
class RunN
{
	unsigned int count_;
public:
	RunN()
		:	count_(0)
	{}
	inline bool next()
	{
		return count_++ < N;
	}
	inline void reset()
	{
		count_ = 0;
	}
	inline void shutdown()
	{
		count_ = N;
	}
	
};

typedef RunN<1> RunOnce;

class RunForever
{	
	bool keepRunning_;
public:	
	RunForever()
		:	keepRunning_(true)
	{}
	inline bool next() { return keepRunning_; }	
	inline void reset() { keepRunning_ = true; }
	inline void shutdown() { keepRunning_ = false; }	
};

// This class allows you to plug a normal class into a thread.
// The plugged class needs to provide a public or proteced
// method loopImpl() which will be called by the thread.
// The thread gets created in PThreadObjectSemantic's
// c'tor and terminated in its d'tor. The policy used
// (template parameter U) determins how often loopImpl() gets called.
template<class T, class U = RunOnce>
class PThreadObjectSemantic : public T, private PThread
{
public:
	// If the thread has terminated when PThreadObjectSemantic's
	// d'tor is called the object will be destroyed. Otherwise
	// the thread is told to shut down via the policy method
	// shutdown(). Note that for PThreadObjectSemantic objects
	// that use the "RunForever" policy you need to invoke
	// the shutdown() method manually with false as parameter
	// to exit the thread.
	virtual ~PThreadObjectSemantic()
	{
		shutdown();
	}
	// Spawn off a thread that runs at least once
	// while the object exists. When the c'tor has
	// been worked off, either a thread is running
	// OR an exception of type CouldNotCreateThread
	// has been thrown.
	PThreadObjectSemantic()	
		:	T()		
		,	isRunning_(false)
	{}
	// Constructor that passes its single argument to T's
	// c'tor.
	template<class X>
	PThreadObjectSemantic(X& arg)	
		:	T(arg)		
		,	isRunning_(false)
	{}
	// c'tor.
	template<class X, class Y>
	PThreadObjectSemantic(X& arg1, Y& arg2)	
		:	T(arg1, arg2)		
		,	isRunning_(false)
	{}

	// c'tor.
	template<class X, class Y, class Z>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3)	
		:	T(arg1, arg2, arg3)		
		,	isRunning_(false)
	{}
	// c'tor.
	template<class X, class Y, class Z, class A>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4)	
		:	T(arg1, arg2, arg3, arg4)		
		,	isRunning_(false)
	{}
	// c'tor.
	template<class X, class Y, class Z, class A, class B>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4, B& arg5)	
		:	T(arg1, arg2, arg3, arg4, arg5)		
		,	isRunning_(false)
	{}
	// c'tor.
	template<class X, class Y, class Z, class A, class B, class C>
	PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4, B& arg5, C& arg6)	
		:	T(arg1, arg2, arg3, arg4, arg5, arg6)		
		,	isRunning_(false)
	{}
	// You may call this method to shut down the 
	// thread before the d'tor is reached.
	// Call this with false as parameter to 
	// terminate the thread as soon as possible
	void shutdown(bool waitTillCompletion = true) 
	{ 
		if(!waitTillCompletion)
			runCondition_.shutdown();

		join();
	}
	// Call this if you wish to rerun the thread
	// You may only call this method if there is
	// no thread currently running. This method
	// may cause a CouldNotCreateThread exception.
	bool start()
	{
		if(isRunning_)
			return false;
		
		startUp();
		return true;
	}
private:
	void threadImpl()
	{
		isRunning_ = true;
		threadStarted_.unlock();		
		
		while(runCondition_.next())
			T::operator()();
		
		isRunning_ = false;
	}
	
	void startUp()
	{
		runCondition_.reset();
		threadStarted_.lock();
		
		if(run() != PT_THREAD_OK)
			throw CouldNotCreateThread();
		
		threadStarted_.lock();
	}
private:
	U runCondition_;
	PMutex threadStarted_;	
	volatile bool isRunning_;
};

PT_NAMESPACE_END

#endif
