/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_UTILITY_H
#define PT_UTILITY_H
#include <portablethreads/config.h>
#include <portablethreads/semaphore.h>
#include <portablethreads/atomic_number.h>
#include <cassert>
#include <cstdio>
#include <vector>

PT_NAMESPACE_BEGIN


// Sleep functions
PT_DLL_EXPORT void pt_second_sleep(unsigned time);
PT_DLL_EXPORT void pt_milli_sleep(unsigned time);
PT_DLL_EXPORT void pt_micro_sleep(unsigned time);
PT_DLL_EXPORT void pt_yield();

// Helper the implements a guard
// To be used in functions that allow only one active thread at a time
// Assume "Lock" is a member of type PMutex or some other class
//
// bool SomeClass::is_empty()
// {
//		PMutexHelper<PMutex> dynunlock(Lock);
//		return SomeValue;
// }
//
// The mutex unlocks when the guards destructor is called
//   
template<class T> 
class PTGuard
{
public:
	// Locks mutex on creation
 	PTGuard(T& mutex) 
		:	mutex_(mutex)
	{
		mutex_.lock();
	}
	
	// unlock mutex when going out of scope
	~PTGuard()
	{
		mutex_.unlock();
	}
private:
	T& mutex_;
};

// Oposite behavior of PTGuard
template<class T> 
class PTAntiGuard
{
public:
	PTAntiGuard(T& mutex) 
		:	mutex_(mutex)
	{
		mutex_.unlock();
	}
	~PTAntiGuard()
	{
		mutex_.lock();
	}
private:
	T& mutex_;
};

#define PMutexHelper PTGuard

// Mersenne Twister 32 bit (pseudo) random numbers, due to: http://www.math.keio.ac.jp/matumoto/emt.html
// This class aims to replace the standard rand() and srand() functions in a mt-safe
// way. While concurrent access to one object still needs to be synchronized, different
// threads may each utilize an different PRandom without interfering with each other.
class PRandom
{
public:
	PT_DLL_EXPORT ~PRandom();
	PT_DLL_EXPORT PRandom(unsigned long seed = 5489UL);
	// Seeds the random number generator.
	// This als resets the generator.
	PT_DLL_EXPORT void seed(unsigned long seed = 5489UL);
	// Return the next random number
	// in the range [-max(long), max(long)]
	inline long rand()
	{
		return static_cast<long>(urand());
	}
	// Return the next random number in the
	// range [0, max(unsigned long)]
	PT_DLL_EXPORT unsigned long urand();
private:
	PRandom(const PRandom&);
	PRandom& operator=(const PRandom&);
private:
	std::vector<unsigned long> mt;
	unsigned long mti;

	PT_DLL_EXPORT static const unsigned long N, M, MATRIX_A, UPPER_MASK, LOWER_MASK;
};

typedef PRandom PTRandom;

// Time stamps and measuring. On UNIX systems this has (should have) micro second precision
// on WIN32 systems precision depends on the hardware but should be better or equivalent to
// UNIX precision.
// PTime also provides a stop clock interface (via start, stop, difference) and time stamps.
class PTime
{
public:
	typedef uint64 time_type;
	PTime()
		:	start_(0)
		,	end_(0)
	{
		assert(frequency_ && "This platform does not support high performance clocks!");
	}
	// Returns the frequency, that how accuratly can be measured.
	inline uint64 frequency() const
	{
		return frequency_;
	}
	// Starts (and resets) the stopclock
	inline void start()
	{
		start_ = end_ = stamp();
	}
	// Stops the stop clock
	inline void stop()
	{
		end_ = stamp();
	}
	// The time elapsed between start() and stop().
	// Divided this value by the return value of frequency()
	// to get the time in seconds.
	// NOTE: Do not use integer division here, the time measured may
	// be well less than one second.
	inline uint64 difference() const
	{
		return end_ - start_;
	}
	// Optain the current time stamp.
	PT_DLL_EXPORT uint64 stamp() const;
private:
	PT_DLL_EXPORT static uint64 calculateFrequency();
private:
	uint64 start_, end_;
	PT_DLL_EXPORT static const uint64 frequency_;
};

class PTBarrier
{
	typedef PAtomicNumber::int_type int_type;
public:
	PTBarrier(unsigned threads)
		:	size_(threads)
		,	counter_(threads)
	{
		assert(threads);
	}
	bool wait()
	{
		const int_type current = --counter_;
		if(current >= 0)
		{
			if(current == 0)
			{
				//std::printf("Unlocking waiters\n");
				for(int_type i = 1; i < size_; ++i)
				{
					sem_.up();
				}
				counter_ = size_;
				return true;
			}
			else
			{
				//std::printf("Blocking on sem\n");
				sem_.down();
			}
		}
		else
		{
			//std::printf("Falling through, counter value: %d\n", (int)current);
		}
		return false;	
	}	
private:
	PTBarrier();
private:	
	const int_type size_;
	PAtomicNumber counter_;
	PSemaphore sem_;
};

typedef PTBarrier PBarrier;

PT_NAMESPACE_END

#endif
