/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifndef WIN32_MUTEX_H
#define WIN32_MUTEX_H
#include <portablethreads/win32/config.h>
#include <cassert>

PT_NAMESPACE_BEGIN
class PMutex
{
	HANDLE mutex_;
	
	inline void init(bool locked)
	{
		mutex_ = locked	? CreateSemaphore(0, 0, 1, 0)
						: CreateSemaphore(0, 1, 1, 0);			
		assert(mutex_ != 0);
	}

	// Mutexes cannot be copied or assigned
	PMutex(const PMutex& o);
  	PMutex& operator=(const PMutex& o);
public:		
	// create a mutex. Default constructor creates unlocked mutex
	PMutex(bool locked = false) 
		:	mutex_(NULL)			
	{
		init(locked);
	}
  	
	~PMutex()
	{
		CloseHandle(mutex_);
	}
	// try to lock the mutex. Returns true on success, false otherwise
	inline bool tryLock()
  	{
      	return WaitForSingleObject(mutex_, 0) == WAIT_OBJECT_0;			
    }
    // lock the mutex. If the mutex is currently locked block till it unlocks
	inline void lock()
  	{
    	while(WaitForSingleObject(mutex_, INFINITE) != WAIT_OBJECT_0);
    }
    // unlock the mutex. This is a noop if the mutex is unlocked
	inline void unlock()
  	{
    	ReleaseSemaphore(mutex_, 1, 0);    		
  	}		
	inline operator HANDLE() const { return mutex_; }
};

class PLightMutex
{
	CRITICAL_SECTION cs_;

	PLightMutex(const PLightMutex&);
	PLightMutex& operator=(const PLightMutex&);
public:
	PLightMutex(bool locked = false)
	{
		InitializeCriticalSection(&cs_);			
		if(locked)
			EnterCriticalSection(&cs_);
	}
	~PLightMutex()
	{
		DeleteCriticalSection(&cs_);
	}

	inline void lock()
  	{
    	EnterCriticalSection(&cs_);
    }

	inline void unlock()
  	{
		LeaveCriticalSection(&cs_);
  	}
#if 0
	inline bool tryLock()
	{
		return TryEnterCriticalSection(&cs_) != 0;
	}
	bool queryState() { return tryLock(); }
#endif
	inline operator CRITICAL_SECTION&() { return cs_; }
	
};

PT_NAMESPACE_END
#endif
