/*
 *	Copyright (c) Januar 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  Einige Beispiele zur Benutzung der Thread-Klassen
 *
 */

#ifdef _MSC_VER
	#pragma warning(disable:4786)
	#pragma warning(disable:4503)
#endif

#include <iostream>
#include <fstream>
#include <string>
#include <cassert>
#include <deque>
#include <list>
#include <ctime>
#include <cstdlib>
#include <cstdio>
#include <set>
#include <portablethreads/mutex.h>
#include <portablethreads/condition.h>
#include <portablethreads/thread.h>
#include <portablethreads/tsallocator.h>
#include <portablethreads/message_queue.h>
#include <portablethreads/utility.h>
#include <portablethreads/atomic_number.h>
#include <portablethreads/lock_free.h>
#ifdef _MSC_VER
	#include <crtdbg.h>
#endif

using namespace std;
using namespace PortableThreads;




#if 0

/*****************************************************************************/

// In diesem Ping-Pong Beispiel wird ein Monitor zur Synchronisation genutzt. 
// Der Vorteil des Monitors gegenber einer Mutex ist das der Monitor potentiell
// performanter ist wenn nur ein Thread gleichzeitig versucht darauf zuzugreifen.
// Bei einer Mutex muss jedes mal der Speicherbus gesperrt werden - beim Monitor
// wird durch atomares dekrementieren eines Zhlers sichergestellt das sich jeweils
// nur ein Thread im Monitor befindet. Versucht ein weiterer Thread den Monitor zu
// betreten, so muss er warten, bis der aktive Thread den Monitor verlsst und dies
// ggf. wartenden Threads signalisiert.


namespace beispiel_8
{
	class PingPong : public PThread
	{
	public:
		PingPong(int id, PMonitor& monitor, volatile bool& ping)
			:	id_(id)
			,	monitor_(monitor)
			,	ping_(ping)
		{}
		void threadImpl()
		{
			for(int i = 0; i < 10; ++i)
			{
				// Versuche den Monitor zu betreten. Falls bereits ein Thread
				// im Monitor ist, warten wir blockierend.
				monitor_.enter();
				cout << id_ << " " << (ping_ ? "ping" : "pong") << endl;
				ping_ = !ping_;
				monitor_.leave();
			}
		}
	private:
		const int id_;
		PMonitor& monitor_;
		volatile bool& ping_;
	};
}

void beispiel8()
{
	cout << "Ping-Pong Beispiel mit 2 Threads die durch einen Monitor synchronisiert werden. ";
	cout << "Press enter to start" << endl;
	cin.get();

	using namespace beispiel_8;

	PMonitor monitor;
	volatile bool ping = true;

	// Drei bzw. zwei oder ein Threads versuchen konkurrierend dPing bzw. Pong auszugeben.
	PingPong p1(1, monitor, ping), p2(2, monitor, ping), p3(3, monitor, ping);
	
	p3.run();
	p2.run();
	p1.run();

	p1.join();
	p2.join();
	p3.join();


	p2.run();
	p1.run();

	p1.join();
	p2.join();

	p1.run();

	p1.join();
}


namespace beispiel_12
{

	typedef vector<unsigned long> ShortList;
	typedef pair<size_t, ShortList > Element;
	typedef PLockFreeQueue<Element> Queue;
	
	class Reader : public PThread
	{
	public:
		Reader(int id, Queue& queue)
			:	id_(id)
			,	shutdown_(false)
			,	queue_(queue)
		{}

		void threadImpl()
		{
			Element e;			
			while(!shutdown_)
			{
				bool didRemove = queue_.popFront(e);
				if(didRemove)
				{
					int ok = e.first == e.second.size();
					if(ok)
					{
						for(size_t i = 0; i < e.second.size(); ++i)
						{
							assert(e.second[i] == e.first);
						}
					}
					assert(ok);
				}
				else
					give();
			}
			std::printf("R %d done\n", id_);
		}
		void shutdown() volatile { shutdown_ = true; }
	private:
		const int id_;
		volatile bool shutdown_;
		Queue& queue_;
	};

	class Writer : public PThread
	{
	public:
		Writer(int id, Queue& queue, int runs)
			:	id_(id)
			,	queue_(queue)
			,	runs_(runs)
		{}
		void threadImpl()
		{
			PRandom r(time(0));
			for(int i = 0; i < runs_; ++i)
			{
				unsigned long n = r.urand() % 100;
				Queue::value_type vt = make_pair(n, ShortList(n, n));
				queue_.pushBack(vt);
				if(i % 5 == 0)
					pt_milli_sleep(50);
			}
		}
	private:
		const int id_;
		Queue& queue_;
		int runs_;
	};
}

/*****************************************************************************/

// Testcode (Reader-Writer) fr Lock-Free Datenstrukturen. Es werden die Queue 
// und der Stack getestet mit jeweils zwei Readern und Writern. 
// Jede Datenstruktur wird einmal mit aktivierten Speichermanagement (fat,
// interner Speicher wird wiederverwendet)
// und einmal ohne (lean, jedes Einfgen zieht eine heap-Allocation nach sich)
// getest.

void beispiel12()
{
	using namespace beispiel_12;
	// queue lean
	if(1)
	{
		Queue queue(true);
		Reader r1(1, queue), r2(2, queue), r3(3, queue), r4(4, queue);

		r1.run();
		r2.run();
		r3.run();
		r4.run();

		Writer w1(1, queue, 20), w2(2, queue, 30), w3(3, queue, 30), w4(4, queue, 20);
		w1.run();
		w2.run();
		w3.run();
		w4.run();

		w1.join();
		w2.join();
		w3.join();
		w4.join();

		std::printf("Writers joined\n");
		r1.shutdown();
		r2.shutdown();
		r3.shutdown();
		r4.shutdown();
		std::printf("Signaled readers shutdown\n");
		r1.join();
		std::printf("R1 joined\n");
		r2.join();
		std::printf("R2 joined\n");
		r3.join();
		std::printf("R3 joined\n");
		r4.join();
		std::printf("R4 joined\n");
		std::printf("Forcing queue d'tor\n");
	}
	std::printf("Queue d'tor went ok\n");

	// queue fat
	if(1)
	{
		Queue queue;
		Reader r1(1, queue), r2(2, queue), r3(3, queue), r4(4, queue);

		r1.run();
		r2.run();
		r3.run();
		r4.run();

		Writer w1(1, queue, 20), w2(2, queue, 30), w3(3, queue, 30), w4(4, queue, 20);
		w1.run();
		w2.run();
		w3.run();
		w4.run();

		w1.join();
		w2.join();
		w3.join();
		w4.join();

		std::printf("Writers joined\n");
		r1.shutdown();
		r2.shutdown();
		r3.shutdown();
		r4.shutdown();
		std::printf("Signaled readers shutdown\n");
		r1.join();
		std::printf("R1 joined\n");
		r2.join();
		std::printf("R2 joined\n");
		r3.join();
		std::printf("R3 joined\n");
		r4.join();
		std::printf("R4 joined\n");
		std::printf("Forcing queue d'tor\n");
	}
	std::printf("Queue d'tor went ok\n");
	
	
}

/*****************************************************************************/

// Testcode fr PAtomicNumber. Der Code soll sicherstellen, dass atomare
// Vernderungen an der Zahl auch wirklich atomar sind.


namespace beispiel_13
{
	class AtomicBomber : public PThread
	{
	public:
		AtomicBomber(PAtomicNumber& n, int runs, bool up)
			:	n_(n)
			,	runs_(runs)
			,	up_(up)
		{}
		void threadImpl()
		{
			PRandom r(time(0));
			for(int i = 0; i < runs_; ++i)
			{
				uint64 n = r.urand();
				if(up_)
				{
					n_.inc(2*n);
					n_.dec(2*n-1);
				}
				else
				{
					n_.dec(2*n);
					n_.inc(2*n-1);
				}
			}
		}
	private:
		PAtomicNumber& n_;
		const int runs_;
		bool up_;
	};

}


void beispiel13()
{
	using namespace beispiel_13;

	PAtomicNumber n;
	AtomicBomber b1(n, 10000000, true), b2(n, 5000000, true), b3(n, 5000000, false), b4(n, 10000000, false);

	b1.run();
	b2.run();
	b3.run();
	b4.run();

	b1.join();
	b2.join();
	b3.join();
	b4.join();

	if(n.get() != 0)
	{
		cout << n.get() << endl;
	}
	assert(n.get() == 0);	
}
/*****************************************************************************/

// Rudimentrer sanity-check fr PAtomicNumber

void beispiel11()
{
	PAtomicNumber n;
	const PAtomicNumber::int_type shouldBe43 = n.inc(43);
	cout << "Return value should be 43, is: " << shouldBe43 << endl;
	cout << "n should be 43, is: " << n.get() << endl;
	assert(shouldBe43 == 43);
	assert(n.get() == 43);
	const PAtomicNumber::int_type shouldBe42 = n.dec(1);
	cout << "Return value should be 42, is: " << shouldBe42 << endl;
	cout << "n should be 42, is: " << n.get() << endl;
	assert(shouldBe42 == 42);
	assert(n.get() == 42);


	n = 0;
	bool ok = n.swap(-1, 0);
	cout << "Swap should succeed, did: " << ok << endl;
	assert(ok);
	ok = n.swap(-2, 0);
	cout << "Swap should NOT succeed, did: " << ok << endl;
	ok = n.swap(-1, -1);
	cout << "Swap should succeed, did: " << ok << endl;
	assert(ok);

	if(1)
	{
		int64 n = 0;
		bool ok = pt_atomic_cas(&n, -1, 0);
		cout << "Swap should succeed, did: " << ok << endl;
		assert(ok);
		assert(n == -1);
		
		ok = pt_atomic_cas(&n, -2, 0);
		cout << "Swap should NOT succeed, did: " << ok << endl;
		assert(!ok);
		assert(n == -1);
		ok = pt_atomic_cas(&n, -1, -1);
		cout << "Swap should succeed, did: " << ok << endl;
		assert(ok);
		assert(n == -1);
	}
}



#endif
