/*
 *	Copyright (c) Januar 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  Einige Beispiele zur Benutzung der Thread-Klassen
 *
 */

#ifdef _MSC_VER
	#pragma warning(disable:4786)
	#pragma warning(disable:4503)
#endif

#include <iostream>
#include <fstream>
#include <string>
#include <cassert>
#include <deque>
#include <list>
#include <ctime>
#include <cstdlib>
#include <cstdio>
#include <set>
#include <portablethreads/mutex.h>
#include <portablethreads/smartpointer.h>
#include <portablethreads/condition.h>
#include <portablethreads/thread.h>
#include <portablethreads/message_queue.h>
#include <portablethreads/utility.h>
#include <portablethreads/atomic_number.h>
#include <portablethreads/lock_free.h>
#include <portablethreads/semaphore.h>
#ifdef _MSC_VER
	#include <crtdbg.h>
#endif

using namespace std;
using namespace PortableThreads;


namespace beispiel_12
{

	typedef vector<unsigned long> ShortList;
	typedef pair<size_t, ShortList > Element;
	typedef PLockFreeQueue<Element> Queue;
	
	class Reader : public PThread
	{
	public:
		Reader(int id, Queue& queue, PAtomicNumber& n)
			:	id_(id)
			,	shutdown_(false)
			,	queue_(queue)
			,	n_(n)
		{}

		void threadImpl()
		{
			Element e;			
			while(!shutdown_)
			{
				bool didRemove = queue_.popFront(e);
				if(didRemove)
				{
					--n_;
					int ok = e.first == e.second.size();
					if(ok)
					{
						for(size_t i = 0; i < e.second.size(); ++i)
						{
							assert(e.second[i] == e.first);
						}
					}
					assert(ok);
				}
				else
					give();
			}
			std::printf("R %d done\n", id_);
		}
		void shutdown() { shutdown_ = true; }
	private:
		const int id_;
		volatile bool shutdown_;
		Queue& queue_;
		PAtomicNumber& n_;
	};

	class Writer : public PThread
	{
	public:
		Writer(int id, Queue& queue, int runs)
			:	id_(id)
			,	queue_(queue)
			,	runs_(runs)
		{}
		void threadImpl()
		{
			PRandom r(time(0));
			for(int i = 0; i < runs_; ++i)
			{
				unsigned long n = r.urand() % 100;
				Queue::value_type vt = make_pair(n, ShortList(n, n));
				queue_.pushBack(vt);
				if(i % 5 == 0)
					pt_milli_sleep(50);
			}
		}
	private:
		const int id_;
		Queue& queue_;
		int runs_;
	};
}

/*****************************************************************************/

// Testcode (Reader-Writer) fr Lock-Free Datenstrukturen. Es werden die Queue 
// und der Stack getestet mit jeweils zwei Readern und Writern. 
// Jede Datenstruktur wird einmal mit aktivierten Speichermanagement (fat,
// interner Speicher wird wiederverwendet)
// und einmal ohne (lean, jedes Einfgen zieht eine heap-Allocation nach sich)
// getest.

void beispiel12()
{
	using namespace beispiel_12;
	
	// queue fat
	if(1)
	{
		Queue queue;
		PAtomicNumber n(100);
		Reader r1(1, queue, n), r2(2, queue, n), r3(3, queue, n), r4(4, queue, n);

		r1.run();
		r2.run();
		r3.run();
		r4.run();

		Writer w1(1, queue, 20), w2(2, queue, 30), w3(3, queue, 30), w4(4, queue, 20);
		w1.run();
		w2.run();
		w3.run();
		w4.run();

		w1.join();
		w2.join();
		w3.join();
		w4.join();

		std::printf("Writers joined\n");
		r1.shutdown();
		r2.shutdown();
		r3.shutdown();
		r4.shutdown();
		std::printf("Signaled readers shutdown\n");
		r1.join();
		std::printf("R1 joined\n");
		r2.join();
		std::printf("R2 joined\n");
		r3.join();
		std::printf("R3 joined\n");
		r4.join();
		std::printf("R4 joined\n");
		Element e;
		for(PAtomicNumber::int_type i = 0; i < n.get(); ++i)
		{
			bool removed = queue.popFront(e);
			assert(removed);
			int ok = e.first == e.second.size();
			if(ok)
			{
				for(size_t i = 0; i < e.second.size(); ++i)
				{
					assert(e.second[i] == e.first);
				}
			}
			assert(ok);
		}
		std::printf("Forcing queue d'tor\n");
	}
	std::printf("Queue d'tor went ok\n");
	
	
}


namespace beispiel_12_1
{
	typedef vector<unsigned long> ShortList;
	typedef pair<size_t, ShortList > Element;
	typedef PLockFreeStack<Element> Stack;
	
	class Reader : public PThread
	{
	public:
		Reader(int id, Stack& s, PAtomicNumber& n)
			:	id_(id)
			,	shutdown_(false)
			,	stack_(s)
			,	n_(n)
		{}
		void threadImpl()
		{
			Element e;
			
			while(!shutdown_)
			{
				bool didRemove = stack_.pop(e);
				if(didRemove)
				{
					--n_;
					int ok = e.first == e.second.size();
					if(ok)
					{
						for(size_t i = 0; i < e.second.size(); ++i)
						{
							assert(e.second[i] == e.first);
						}
					}
					assert(ok);
					
				}
				else
					give();
				
			}
			std::printf("R %d done\n", id_);
		}
		void shutdown() volatile { shutdown_ = true; }
	private:
		const int id_;
		volatile bool shutdown_;
		Stack& stack_;
		PAtomicNumber& n_;
	};

	class Writer : public PThread
	{
	public:
		Writer(int id, Stack& s, int runs, PAtomicNumber& n)
			:	id_(id)
			,	stack_(s)
			,	runs_(runs)
			,	n_(n)
		{}
		void threadImpl()
		{
			PRandom r(time(0));
			for(int i = 0; i < runs_; ++i)
			{
				unsigned long n = r.urand() % 100;
				Stack::value_type vt = make_pair(n, ShortList(n, n));
				stack_.push(vt);
				++n_;
				if(i % 5 == 0)
					pt_milli_sleep(50);
			}
		}
	private:
		const int id_;
		Stack& stack_;
		int runs_;
		PAtomicNumber& n_;
	};
}

void beispiel12_1()
{
	using namespace beispiel_12_1;
	// stack_ lean
	{
		Stack stack_(true);
		PAtomicNumber n;
		Reader r1(1, stack_, n), r2(2, stack_, n), r3(3, stack_, n), r4(4, stack_, n);

		r1.run();
		r2.run();
		r3.run();
		r4.run();

		Writer w1(1, stack_, 20, n), w2(2, stack_, 30, n), w3(3, stack_, 30, n), w4(4, stack_, 20, n);
		w1.run();
		w2.run();
		w3.run();
		w4.run();

		w1.join();
		w2.join();
		w3.join();
		w4.join();

		std::printf("Writers joined\n");
		r1.shutdown();
		r2.shutdown();
		r3.shutdown();
		r4.shutdown();
		std::printf("Signaled readers shutdown\n");
		r1.join();
		std::printf("R1 joined\n");
		r2.join();
		std::printf("R2 joined\n");
		r3.join();
		std::printf("R3 joined\n");
		r4.join();
		std::printf("R4 joined\n");
		Element e;
		for(PAtomicNumber::int_type i = 0; i < n.get(); ++i)
		{
			bool removed = stack_.pop(e);
			assert(removed);
			int ok = e.first == e.second.size();
			if(ok)
			{
				for(size_t i = 0; i < e.second.size(); ++i)
				{
					assert(e.second[i] == e.first);
				}
			}
			assert(ok);
		}
		std::printf("Forcing stack d'tor\n");
	}
	std::printf("Stack d'tor went ok\n");

	// stack_ fat
	{
		Stack stack_(false);
		PAtomicNumber n;
		Reader r1(1, stack_, n), r2(2, stack_, n), r3(3, stack_, n), r4(4, stack_, n);

		r1.run();
		r2.run();
		r3.run();
		r4.run();

		Writer w1(1, stack_, 20, n), w2(2, stack_, 30, n), w3(3, stack_, 30, n), w4(4, stack_, 20, n);
		w1.run();
		w2.run();
		w3.run();
		w4.run();

		w1.join();
		w2.join();
		w3.join();
		w4.join();

		std::printf("Writers joined\n");
		r1.shutdown();
		r2.shutdown();
		r3.shutdown();
		r4.shutdown();
		std::printf("Signaled readers shutdown\n");
		r1.join();
		std::printf("R1 joined\n");
		r2.join();
		std::printf("R2 joined\n");
		r3.join();
		std::printf("R3 joined\n");
		r4.join();
		std::printf("R4 joined\n");
		Element e;
		for(PAtomicNumber::int_type i = 0; i < n.get(); ++i)
		{
			bool removed = stack_.pop(e);
			assert(removed);
			int ok = e.first == e.second.size();
			if(ok)
			{
				for(size_t i = 0; i < e.second.size(); ++i)
				{
					assert(e.second[i] == e.first);
				}
			}
			assert(ok);
		}
		std::printf("Forcing stack d'tor\n");
	}
	std::printf("Stack d'tor went ok\n");
	
	
}

namespace beispiel_14
{
	using namespace PortableThreads::PTPrivate;
	class StressCAS : public PThread
	{
	public:
		typedef std::vector<PTPointerCAS::int_t> IdList;
		StressCAS(PBarrier& barrier, int id, int runs, PTPointerCAS& cas)
			:	barrier_(barrier)
			,	id_(id)
			,	runs_(runs)
			,	cas2_(cas)
		{}
	private:
		void threadImpl()
		{
			barrier_.wait();
			PTPointerCAS::token_t current, old;
			PTPointerCAS::int_t nv;
			for(int i = 0; i < runs_; ++i)
			{
				int tries = 0;
				do
				{
					if(tries)
					{
						printf("%d trying for the %d. time\n", id_, tries+1);
						old = current;
					}
					current = cas2_.get();
					assert(current.value() >= 0 && current.value() % 8 == 0);
					if(tries)
					{
						if(current.count() <= old.count())
						{
							printf("Current count should be larger than old count: %d:%d\n", current.count(), old.count());

						}
						assert(current.count() > old.count());
					}
					nv = 8*(i);
				
					++tries;					
				}
				while(!cas2_.cas(nv, current));
				ids_.push_back(current.count());
				printf("%d changed from %ld:%ld to %ld\n", id_, 
					current.value(), current.count(), nv);
			}
		}
	public:
		const IdList& ids() const { return ids_; }
	private:
		PBarrier& barrier_;
		const int id_;
		const int runs_;
		PTPointerCAS& cas2_;
		IdList ids_;
	};
}

void beispiel14()
{
	using namespace beispiel_14;

	const int n = 8;
	const int runs = 20;
	PBarrier barrier(n);
	PTPointerCAS value;
	std::vector< StackPtr<StressCAS> > threads(n);
	typedef std::set<PTPointerCAS::int_t> IdSet;
	IdSet ids;
	for(int i = 0; i < n; ++i)
	{
		threads[i] = new StressCAS(barrier, i, runs, value);
		threads[i]->run();
	}
	for(int i = 0; i < n; ++i)
	{
		threads[i]->join();
		for(StressCAS::IdList::const_iterator it = threads[i]->ids().begin();
			it != threads[i]->ids().end(); ++it)
		{
			ids.insert(*it);
		}
	}
	if(value.get().count() < n*runs)
	{
		cerr << "error! Expected at least: " << (n*runs) << ", is: " << value.get().count() << endl;
	}
	assert(value.get().count() >= n*runs);
	if(ids.size() != n*runs)
	{
		cerr << "error! Expected set size: " << (n*runs) << ", is: " << ids.size() << endl;
	}
	assert(ids.size() == n*runs);
	cout << "OK" << endl;
}

namespace beispiel_15
{
	
	inline int32 high(int64 large)
	{
		return (int32)(large >> 32);
	}
	inline int32 low(int64 large)
	{
		return (int32)large;
	}

	static inline int64 multiplex(int32 value, int32 count)
	{
		const int32 buf[2] = { value, count };
		return *((int64*)buf);
	}


	static inline uint64 umultiplex(uint32 value, uint32 count)
	{
		const uint32 buf[2] = { value, count };
		return *((uint64*)buf);
	}

	static inline void assign(int64& value, const int32* buf)
	{
		value = *(int64*)buf;
	}

	static inline void uassign(uint64& value, const uint32* buf)
	{
		value = *(uint64*)buf;
	}

	static inline void assign(int32* buf , int64 value)
	{
		*(int64*)buf = value;
	}

	static inline void uassign(uint32* buf , uint64 value)
	{
		*(uint64*)buf = value;
	}

	union union_int
	{
		int64 large_;
		struct { int32 high_, low_; } parts_;
	};

	union union_uint
	{
		uint64 large_;
		struct { uint32 high_, low_; } parts_;
	};

}

void beispiel15()
{
	int64 n = 1<<6;
	assert(n == 64);
	n <<= 19;
	n++;

	cout << "Count should be 1, is: " << (n & (((int64)1<<19)-1)) << endl;
	cout << "Left shift 16, now: " << (n >> 16) << endl;
	cout << "Left shift 19, now: " << (n >> 19) << endl;
	cout << "Value should be 64, is: " << ((n >> 19) & ~(((int64)1<<3)-1)) << endl;

	// Attention: All values passed to PTPointerCAS must not have any lowerorder bits
	// that are smaller than 7!
	PTPointerCAS ptc;
	PTPointerCAS::token_t current, old;

	ptc.assign(64);

	current = ptc.get();
	cout << "Value should be 64, is: " << current.value() << endl;
	assert(current.value() == 64);
	
	bool ret = ptc.cas(8, current);
	assert(ret);
	current = ptc.get();
	cout << "Value should be 8, is: " << current.value() << endl;
	assert(current.value() == 8);

	ptc.assign(0);

	for(int i = 0; i < 10; ++i)
	{
		old = current;
		current = ptc.get();
		if(current.value() != (i*8))
		{
			cout << "i: " << i << ". value should be " << (i*8) << ", is " << current.value() << endl;
		}
		assert(current.value() == (i*8));
		if(i)
		{
			
			assert(current.count() == old.count() + 1);			
		}
		bool ok = ptc.cas(8*(i+1), current);
		if(!ok)
		{
			cout << "Failed with i=" << i << endl;
		}
		assert(ok);
	}
}

void beispiel16()
{
	PAtomicNumber n(-42);
	assert(n.get() == -42);
	n.inc(42);
	assert(n.get() == 0);
	n.inc(42);
	PAtomicNumber::int_type res = n++;
	assert(res == 42);
	assert(n.get() == 43);
	
	n = -33;
	
	assert(n.get() == -33);
	res = --n;
	assert(res == -34);
	assert(n.get() == -34);

}

void beispiel17()
{
		volatile uint8 lock = 0;
	bool ret = pt_atomic_set_lock(&lock);
	assert(lock != 0);
	assert(ret);
	pt_atomic_clear_lock(&lock);
	assert(lock == 0);
	pt_atomic_clear_lock(&lock);
	assert(lock == 0);
	ret = pt_atomic_set_lock(&lock);
	assert(lock != 0);
	assert(ret);
	ret = pt_atomic_set_lock(&lock);
	assert(lock != 0);
	assert(!ret);
}

template<class T>
struct bits
{
	T t_;
	bits(T t = 0)
		:	t_(t)
	{}
};

template<class T>
ostream& operator<<(ostream& os, bits<T>& b)
{
	for(unsigned i = 8*sizeof(T)-1; i > 0; --i)
	{
		os << ((b.t_ & (1<<i)) == 0 ? '0' : '1');
	}
	return os;
}

void ultraSPARC()
{
	// UltraSPARC PointerCAS simulation

	const int64 MSB_MASK = static_cast<int64>(1)<<63;
	const int64 HIGH_ONES = (((int64)1<<21)-1) << 44;
	const int64 LOW_ONES = ~HIGH_ONES;

	cout << ((((int64)-1) & MSB_MASK)  == 0) << endl;
	cout << ((((int64)1<<63) & MSB_MASK)  == 0) << endl;
	cout << ((((int64)0) & MSB_MASK)  == 0) << endl;
	
	int64 low = 42;
	int64 high = low | HIGH_ONES;

	int64 count = ((int64)33) << 44;

	int64 mux = low ^ count;

	int64 counter = (mux & MSB_MASK) == 0 ? mux >> 44 : (~mux) >> 44;
	int64 value = (mux & MSB_MASK) == 0 ? mux & LOW_ONES : mux | HIGH_ONES;
	cout << counter << ", " << value << endl;

	mux = high ^ count;
	counter = (mux & MSB_MASK) == 0 ? mux >> 44 : (~mux) >> 44;
	value = (mux & MSB_MASK) == 0 ? mux & LOW_ONES : mux | HIGH_ONES;
	cout << counter << ", " << value << ". Same as high: " << (value == high) << endl;


}

int main()
{
#ifdef _MSC_VER
	if(1)
	{
		_CrtSetDbgFlag(_CrtSetDbgFlag(_CRTDBG_REPORT_FLAG) |
					_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF |
		_CRTDBG_CHECK_ALWAYS_DF);
		_CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_FILE );
		_CrtSetReportFile( _CRT_WARN, _CRTDBG_FILE_STDERR );
		_CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_FILE );
		_CrtSetReportFile( _CRT_ERROR, _CRTDBG_FILE_STDERR );
		_CrtSetReportMode( _CRT_ASSERT, _CRTDBG_MODE_FILE );
		_CrtSetReportFile( _CRT_ASSERT, _CRTDBG_FILE_STDERR );
	}
#endif
	
	
	
	/*
	beispiel14();
	beispiel12_1();
	*/
	//beispiel12();

	beispiel14();

	beispiel15();
	beispiel16();

	
	return 0;
}

