/*	$Id: single_threaded_core.cpp 2058 2005-06-29 07:52:37Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <cassert>
#include <nop_stream.h>
#include <platypus/types.h>
#include <platypus/types_factory.h>
#include <platypus/factories/core_factory.h>
#include <platypus/factories/registration.h>
#include <platypus/options.h>
#include <platypus/cores/kernel.h>
#include <platypus/cores/single_threaded_core.h>

using namespace std;

namespace Platypus
{
	/************************************************************************/
	/* STCallbackAdaptor                                                   */
	/************************************************************************/
	class STCallbackAdaptor	: public KernelCallback
	{
	public:
		STCallbackAdaptor();
		void program(ProgramInterface& prog);
		void distribution(CoreDistributionCallback& callback);
		void factory(const PlatypusTypesFactory& factory);
		void output(NopStream& str);
		void options(const PlatypusOptions& values);
	private:
		bool shutdown();
		bool delegate();		
		void delegate(const DelegatableChoice& dc);
		bool branch();
		void branch(const DelegatableChoice& dc);
		Expander* expander(const DelegatableChoice& dc = DelegatableChoice()); 
		DelegatableChoicePolicy::PolicyBase* delegatableChoicePolicy();
		ProbingPolicy::PolicyBase* probingPolicy();
		void answerSet(const PartialAssignment& pa);
		void markConflict();
		void markAnswerSet();
		void markBacktrack();
		void markExpanderInitialization();
		int processId() const;
		int threadId() const;
		bool silent() const;
		ProbingManager* probingManager();
		NopStream* output();
		size_t atoms() const;
		DelegationHeuristicInterface* delegationHeuristic();
	private:
		CoreDistributionCallback* traits_;
		ProgramInterface* program_;
		const PlatypusTypesFactory* factory_;
		const PlatypusOptions* options_;
		NopStream* os_;
	};

	STCallbackAdaptor::STCallbackAdaptor()
		:	traits_(0)
		,	program_(0)
		,	factory_(0)
		,	options_(0)
		,	os_(0)
	{}
	void STCallbackAdaptor::program(ProgramInterface& prog)
	{
		program_ = &prog;
	}
	void STCallbackAdaptor::distribution(CoreDistributionCallback& callback)
	{
		traits_ = &callback;
	}
	void STCallbackAdaptor::factory(const PlatypusTypesFactory& factory)
	{
		factory_ = &factory;
	}
	void STCallbackAdaptor::output(NopStream& str)
	{
		os_ = &str;
	}
	void STCallbackAdaptor::options(const PlatypusOptions& values)
	{
		options_ = &values;
	}
	bool STCallbackAdaptor::delegate()
	{
		return traits_->needDelegatableChoice();
	}
	bool STCallbackAdaptor::shutdown()
	{
		return traits_->shutdown();
	}
	void STCallbackAdaptor::delegate(const DelegatableChoice& dc)
	{
		traits_->delegate(dc);
	}
	Expander* STCallbackAdaptor::expander(const DelegatableChoice& dc)
	{
		std::auto_ptr<Expander> expander(factory_->expanderFactory().create(options_->expander(), *program_, dc));
		expander->setChoicePolicy(factory_->choicePolicyFactory().create(options_->localChoicePolicy()));
		expander->setManualChoice(options_->manualChoices());
		
		return expander.release();
	}
	
	DelegatableChoicePolicy::PolicyBase* STCallbackAdaptor::delegatableChoicePolicy()
	{
		return factory_->delegatableChoicePolicyFactory().create(options_->delegatableChoicePolicy());
	}
	ProbingPolicy::PolicyBase* STCallbackAdaptor::probingPolicy()
	{
		return factory_->probingPolicyFactory().create(options_->probingPolicy());
	}
	void STCallbackAdaptor::answerSet(const PartialAssignment& pa)
	{
		assert(!options_->silent());
		traits_->answerSet(pa);
	}
	void STCallbackAdaptor::markConflict()
	{
		traits_->incConflicts();
	}
	void STCallbackAdaptor::markAnswerSet()
	{
		traits_->incAnswerSets();
	}
	void STCallbackAdaptor::markBacktrack()
	{
		traits_->incBacktracks();
	}
	void STCallbackAdaptor::markExpanderInitialization()
	{
		traits_->incExpanderInitializations();
	}
	int STCallbackAdaptor::processId() const
	{
		return traits_->id();
	}
	int STCallbackAdaptor::threadId() const
	{
		return 1;
	}
	bool STCallbackAdaptor::silent() const
	{
		return options_->silent();
	}
	ProbingManager* STCallbackAdaptor::probingManager()
	{
		return new ProbingManager(
			options_->probingInitialCutoff(),
			options_->probingSteps(),
			options_->probingBestN(),
			options_->probing());
	}
	NopStream* STCallbackAdaptor::output()
	{
		return os_;
	}
	size_t STCallbackAdaptor::atoms() const
	{
		return program_->numberOfAtoms();
	}
	DelegationHeuristicInterface* STCallbackAdaptor::delegationHeuristic()
	{
		return new DelegateIfPercentageExceeded(options_->percentage());
	}
	bool STCallbackAdaptor::branch()
	{
		return false;
	}
	void STCallbackAdaptor::branch(const DelegatableChoice& dc)
	{
		assert(false && "Branching is for local distribution only, no point in single threaded core!");
	}

	/************************************************************************/
	/* SingleThreadedCore                                                   */
	/************************************************************************/

	const std::string SingleThreadedCore::type_("single-threaded");

	CoreBase* SingleThreadedCore::create()
	{
		return new SingleThreadedCore();
	}
	SingleThreadedCore::~SingleThreadedCore()
	{}
	SingleThreadedCore::SingleThreadedCore()
		:	adaptor_(new STCallbackAdaptor)
		,	kernel_(new Kernel(adaptor_.get()))
		,	traits_(0)
		,	os_(0)
	{}

	void SingleThreadedCore::program(ProgramInterface& prog)
	{
		adaptor_->program(prog);
	}
	void SingleThreadedCore::distribution(CoreDistributionCallback& callback)
	{
		traits_ = &callback;
		adaptor_->distribution(callback);
	}
	void SingleThreadedCore::factory(const PlatypusTypesFactory& fac)
	{
		adaptor_->factory(fac);
	}
	void SingleThreadedCore::output(NopStream& str)
	{
		os_ = &str;
		adaptor_->output(str);
	}
	void SingleThreadedCore::options(const PlatypusOptions& values)
	{
		adaptor_->options(values);
	}
	void SingleThreadedCore::setup()
	{
		kernel_->configure();
	}

	const std::string& SingleThreadedCore::type() const 
	{ 
		return type_; 
	}

	void SingleThreadedCore::run()
	{
		bool filedForDC = false;
		while(!traits_->shutdown())
		{
			if(!filedForDC)
			{
				filedForDC = true;
				traits_->fileDelegatableChoiceRequest();				
			}
			else
			{
				if(traits_->hasDelegatableChoice())
				{
					filedForDC = false;
					const DelegatableChoice initializer(traits_->delegatableChoice());
					(*os_) << "P" << traits_->id() << " " << "got dc " << initializer << " from traits\n";
					kernel_->run(initializer);
				}
			}
		}
		(*os_) << "P" << traits_->id() << " " << "Leavind expand loop\n";
	}

	template<>
	void factoryRegistration<SingleThreadedCore>()
	{
		CoreFactory::instance().add("st", SingleThreadedCore::create, "Use single-threaded core");
	}
}
