#include <iostream>
#include <string>
#include <cstring>
#include <sstream>
#include <stdexcept>
#include "dirstream.h"
#include "program_options.h"
#include "value.h"
#include "uname.h"
#include "path.h"
#include "config.h"

using namespace std;
using namespace ProgramOptions;

unsigned WIDTH = 32;
const char* const LINUX = "Linux";
const char* const SUNOS = "SunOS";
const char* const MINGW32 = "MINGW32";
string lib(lib_path()), include(include_path());

utsname system_description;

bool contains(const std::string& haystack, const char* needle);
bool contains(const char* haystack, const char* needle);
void setPaths(const OptionValues& values);
void setMode(const OptionValues& values);

void libs(bool own, bool shared, std::ostream& os, const utsname& system_description);
void defines(std::ostream& os, const utsname& system_description);
void cxxflags(std::ostream& os, const utsname& system_description);
void ldflags(std::ostream& os, const utsname& system_description);


int main(int argc, char** argv)
{
	OptionGroup options("This is the PortableThreads configuration utility.\nIt provides command line options to compile and link with PortableThreads.");
	options.addOptions()
		("help,h", bool_switch()->defaultValue(false), "print this")
		("defines", bool_switch()->defaultValue(false), "print defines to compile")
		("cxxflags", bool_switch()->defaultValue(false), "print flags for g++ to compile")
		("ldflags", bool_switch()->defaultValue(false), "print flags for g++ to link")
		("libs", bool_switch()->defaultValue(false), "print libs to link with")
		("lib-dir", value<string>(), "override default path for libraries")
		("include-dir", value<string>(), "override default path for header files")
		("no-own-libs", bool_switch()->defaultValue(false), "just put non-PortableThreads libraries with --libs")
		("shared", bool_switch()->defaultValue(false), "use shared libraries")
		("mode", value<string>()->defaultValue("detect"), "use 32 or 64 bit mode, default: detect")
	;
	
	try
	{
		OptionValues values(parseCommandLine(argc, argv, options, true));
		if(option_as<bool>(values, "help"))
		{
			cout << options << endl;
			return 0;
		}

		// gather system information
		uname(&system_description);

		setPaths(values);
		setMode(values);

		

		ostringstream s;
		if(option_as<bool>(values, "defines"))
			defines(s, system_description);

		if(option_as<bool>(values, "cxxflags"))
			cxxflags(s, system_description);

		if(option_as<bool>(values, "ldflags"))
			ldflags(s, system_description);

		if(option_as<bool>(values, "libs"))
			libs(!option_as<bool>(values, "no-own-libs"), option_as<bool>(values, "shared"),
				s, system_description);

		cout << s.str() << endl;;
	}
	catch(const exception& e)
	{
		cout << e.what() << endl;
		return 1;
	}
	catch(...)
	{
		cout << "unknown error, exiting" << endl;
		return 1;
	}

	return 0;
}

void ldflags(std::ostream& os, const utsname& system_description)
{
	
}




void libs(bool own, bool shared, std::ostream& os, const utsname& system_description)
{
	if(own)
	{
		// when using shared lib print -L<libdir> -lportablethreads,
		// otherwise print <libdir>/libportablethreads.a
		if(shared)
		{
			if(contains(system_description.sysname, MINGW32))
			{
				os << lib << '/' << "libportablethreads.dll.a ";
			}
			else
			{
				os << "-L" << lib << " -lportablethreads ";
			}
		}
		else
		{
			os << lib << '/' << "libportablethreads.a ";
		}
	}

	if(contains(system_description.sysname, SUNOS))
	{
		os << "-lpthread -lrt ";
	}
	else if(contains(system_description.sysname, LINUX))
	{
		os << "-lpthread -lrt ";
	}
	else if(contains(system_description.sysname, MINGW32))
	{
		os << "-lwinmm ";
	}
}


void defines(std::ostream& os, const utsname& system_description)
{
	// common
	os << "-D_REENTRANT -DPT_NATIVE_WORD=" << WIDTH << " ";

	// os
	if(contains(system_description.sysname, SUNOS))
	{
		os << "-DPT_SOLARIS ";
	}
	if(contains(system_description.sysname, LINUX))
	{
		os << "-DPT_LINUX ";
	}
	else if(contains(system_description.sysname, MINGW32))
	{
		os << "-DPT_WINDOWS ";
	}

	// arch
	if(contains(system_description.machine, "sparc"))
	{
		os << "-DPT_SPARC ";
	}
	else if(contains(system_description.machine, "i586") || /* Linux box on x86 */
			contains(system_description.machine, "i686") ||
			contains(system_description.machine, "x86_64"))
	{
		os << "-DPT_X86 ";
	}
	else if(contains(system_description.machine, "i86pc")) /* Solaris on x86 */
	{
		os << "-DPT_X86 ";
	}
}

void cxxflags(std::ostream& os, const utsname& system_description)
{
	// header directory
	os << "-I" << include << " -m" << WIDTH << " ";

	// tell compiler to use pthread lib
	if(contains(system_description.sysname, SUNOS))
	{
		os  << "-pthreads ";		
	}
	else if(contains(system_description.sysname, LINUX))
	{
		os << "-pthread ";
	}
}

void setPaths(const OptionValues& values)
{
	if(values.count("lib-dir"))
	{	
		lib = option_as<string>(values, "lib-dir");
		if(!dirstr::directory_exists(lib))
			throw std::runtime_error("value for option --lib-dir must be a directory!");
	}
	if(values.count("include-dir"))
	{	
		include = option_as<string>(values, "include-dir");
		if(!dirstr::directory_exists(include))
			throw std::runtime_error("value for option --include-dir must be a directory!");
	}	
}

void setMode(const OptionValues& values)
{
	// by default 32 bit mode is used
	const string mode = option_as<string>(values, "mode");
	if(mode == "64")
	{
		WIDTH = 64;
	}
	else if(mode == "detect")
	{
		// detect a 64 bit machine		
		if(	contains(system_description.machine, "sun4u") || // UltraSPARC 
			contains(system_description.machine, "x86_64")) // AMD64 or Intel64
		{
			WIDTH = 64;
		}
	}
}

bool contains(const std::string& haystack, const char* needle)
{
	return haystack.find(needle) != std::string::npos;
}

bool contains(const char* haystack, const char* needle)
{
	return strstr(haystack, needle) != NULL;
}

