/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         *
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         *
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <operators/backward_prop.h>
#include <graph.h>
#include <body_node.h>
#include <head_node.h>
#ifdef KDEVELOP
  #include "functors.h"
  #include "algorithms.h"
#else
  #include "../detail/functors.h"
  #include "../detail/algorithms.h"
#endif
#include <cassert>
#include <algorithm>
#include <iterator>
#if defined (_MSC_VER) && _MSC_VER <= 1300
#define for if(0); else for
#endif

namespace NS_NOMORE {

namespace {
	struct CanBeIgnored {
		bool operator()(BodyNode* n) const {
			return n->getColor() == Color::none && n->canBeIgnored();
		}
	};
}

BackwardPropagator::BackwardPropagator(Graph& grp)
  : GraphOperator(BackwardPropagator::getOpName(), grp)
	, jumping_(true)
	, ignore_(true)
	, graphConstructed_(grp.isConstructionFinished()) {

	event::ChannelManager& m = getGraph().getEventManager();
	m.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);
	m.getChannelFor(event::Event<BodyNodeColoredChoicePoint>()).connect(*this);
	m.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).connect(*this);
	m.getChannelFor(event::Event<GraphConstructionFinished>()).connect(*this);
}

void BackwardPropagator::disableJumping() {
  jumping_ = false;
}
void BackwardPropagator::disableIgnore() {
  ignore_ = false;
}

///////////////////////////////////////////////////////////////////////////////
// backward propagation
///////////////////////////////////////////////////////////////////////////////
bool BackwardPropagator::execute() {
	bool keepRunning = false;
  bool changed = false;

  do {
    keepRunning =  handleIgnoreNodes();
		keepRunning |= backpropagateHeadNodes();
    keepRunning |= backpropagateBodyNodes();

    changed |= keepRunning;

  } while(keepRunning);

  return changed;
}

bool BackwardPropagator::handleIgnoreNodes() {
	CanBeIgnored canBeIgnored;
	bool changed = false;
	while (!bodyNodesToIgnore_.empty()) {
		BodyNode* n = bodyNodesToIgnore_.front();
		bodyNodesToIgnore_.pop();

		if (canBeIgnored(n))
			changed |= getGraph().color(*n, Color::ignore);
	}
	return changed;
}

bool BackwardPropagator::backpropagateBodyNodes() {
	bool changed = false;
	while (!bodyNodes_.empty()) {
		BodyNode* n = bodyNodes_.front();
		bodyNodes_.pop();
		changed |= backpropagate(*n);
	}
	return changed;
}

bool BackwardPropagator::backpropagateHeadNodes() {
	bool changed = false;
	while (!headNodes_.empty()) {
		HeadNode* n = headNodes_.front();
		headNodes_.pop();
		changed |= backpropagate(*n);
	}
	return changed;
}

bool BackwardPropagator::backpropagate(BodyNode& b) {
	if (HasColor(Color::plus | Color::weak_plus)(&b)) {
		bool changed = false;
    BodyNode::HeadNodeIterator end = b.zeroPredecessorsEnd();
		for (BodyNode::HeadNodeIterator it = b.zeroPredecessorsBegin(); it != end; ++it) {
			if ( (*it)->getColor() == Color::none) {
				changed |= getGraph().color(**it, (*it)->recursiveSupportColor());
				headNodes_.push(*it);
			}
		}
		end = b.onePredecessorsEnd();
		for (BodyNode::HeadNodeIterator it = b.onePredecessorsBegin(); it != end; ++it) {
			if ( (*it)->getColor() == Color::none) {
				changed |= getGraph().color(**it, Color::minus);
				headNodes_.push(*it);
			}
		}
		return changed;
	}
	else if (b.getColor() == Color::minus) {
		if (b.isUnblocked() && b.countZeroPredecessorsNotPlusOrWeakPlus() == 1) {
			BodyNode::HeadNodeIterator it = std::find_if(b.zeroPredecessorsBegin(),
				b.zeroPredecessorsEnd(), HasColor(Color::none));
			if (it != b.zeroPredecessorsEnd()) {
				getGraph().color(**it, Color::minus);
				headNodes_.push(*it);
				return true;
			}
		}
		else if (b.isWeakSupported() && b.countOnePredecessorsNotMinus() == 1) {
			BodyNode::HeadNodeIterator it = std::find_if(b.onePredecessorsBegin(),
				b.onePredecessorsEnd(), HasColor(Color::none));
			if (it != b.onePredecessorsEnd()) {
				getGraph().color(**it, (*it)->recursiveSupportColor());
				headNodes_.push(*it);
				return true;
			}
		}
	}
	return false;
}

bool BackwardPropagator::backpropagate(HeadNode& h) {
	bool changed = false;
	if (HasColor(Color::plus | Color::weak_plus)(&h)
		&& h.countPredecessorsNotMinus() == 1) {

		HeadNode::BodyNodeIterator it = std::find_if(h.predecessorsBegin(),
			h.predecessorsEnd(), HasColor(Color::none | Color::ignore));
		if (it != h.predecessorsEnd()) {
			changed = getGraph().color(**it, (*it)->recursiveSupportColor());
			bodyNodes_.push(*it);
		}

	}
	else if (h.getColor() == Color::minus) {

		HeadNode::BodyNodeIterator end = h.predecessorsEnd();
		for (HeadNode::BodyNodeIterator it = h.predecessorsBegin(); it != end; ++it) {
			if (HasColor(Color::none | Color::ignore)(*it)) {
				changed |= getGraph().color(**it, Color::minus);
				bodyNodes_.push(*it);
			}
		}
	}
	return changed;
}
///////////////////////////////////////////////////////////////////////////////
// event handlers
///////////////////////////////////////////////////////////////////////////////
void BackwardPropagator::handle(const GraphConstructionFinished& e) {
	assert(e.theGraph_ == &getGraph());
	graphConstructed_ = true;
	event::ChannelManager& m = getGraph().getEventManager();
	if (!jumping_) {
		m.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
		if (!ignore_)
			m.getChannelFor(event::Event<HeadNodeColored>()).disconnect(*this);
	}
}

// choice points are always backpropagated.
void BackwardPropagator::handle(const HeadNodeColoredChoicePoint& e) {
	headNodes_.push(e.node_);
  handle(HeadNodeColored(e.node_, e.oldColor_));
}

void BackwardPropagator::handle(const BodyNodeColoredChoicePoint& e) {
	bodyNodes_.push(e.node_);
  handle(BodyNodeColored(e.node_, e.oldColor_));
}


// if a body node changes its color to minus successors that are
// either plus or weak_plus are backpropagated.
// Only called if in jumping-mode
void BackwardPropagator::handle(const BodyNodeColored& e) {

	if (e.node_->getColor() == Color::minus) {

		DETAIL::copy_if(e.node_->successorsBegin(), e.node_->successorsEnd(),
			DETAIL::pushInserter(headNodes_), HasColor(Color::plus | Color::weak_plus));

	}
}

void BackwardPropagator::handle(const HeadNodeColored& e) {
	if (!graphConstructed_)
		headNodes_.push(e.node_);

	if (ignore_) {
		DETAIL::copy_if(e.node_->predecessorsBegin(), e.node_->predecessorsEnd(),
			DETAIL::pushInserter(bodyNodesToIgnore_), CanBeIgnored());

	}
	// handle jumping
	Color::ColorValue newColor = e.node_->getColor();
	if (newColor == Color::plus || newColor == Color::weak_plus) {

		DETAIL::copy_if(e.node_->zeroSuccessorsBegin(), e.node_->zeroSuccessorsEnd(),
			DETAIL::pushInserter(bodyNodes_), HasColor(Color::minus));

	}
	else if (newColor == Color::minus) {

		DETAIL::copy_if(e.node_->oneSuccessorsBegin(), e.node_->oneSuccessorsEnd(),
			DETAIL::pushInserter(bodyNodes_), HasColor(Color::minus));

	}

}

void BackwardPropagator::postExecute() {
  while(!headNodes_.empty()) { headNodes_.pop(); }
  while(!bodyNodes_.empty()) { bodyNodes_.pop(); }
  while(!bodyNodesToIgnore_.empty()) { bodyNodesToIgnore_.pop(); }
  graphConstructed_ = false;
}

bool BackwardPropagator::validate(const OpSet&) const {
  return true;
}

}
