#include <nomore/nomore_expander.h>
#include <nomore/nomore_program.h>
#include <nomore/nomore_program_impl.h>
#include <nomore/libnomore/include/colorerror.h>
#include <nomore/libnomore/include/graph.h>
#include <nomore/libnomore/include/operators/forward_prop.h>
#include <nomore/libnomore/include/operators/backward_prop.h>
#include <nomore/libnomore/include/operators/unfounded_set.h>
#include <nomore/libnomore/include/operators/sequence.h>
#include <nomore/libnomore/include/operators/hybrid_lookahead.h>
#include <nomore/libnomore/include/operators/select_supported.h>
#include <nomore/libnomore/include/operators/select_uncolored.h>
#include <nomore/libnomore/include/heuristics/hybrid_lookahead_heuristic.h>

#include <platypus/exceptions.h>
#include <platypus/factories/expander_factory.h>
#include <platypus/factories/registration.h>

#include <cassert>
#include <deque>
#include <utility>
#include <algorithm>
using namespace NS_NOMORE;

namespace Platypus { namespace Nomore {

///////////////////////////////////////////////////////////////////////////////
// class Expander::Impl
//
// Implementation of the nomore expander
//
// Note: The expander currently uses the propagation operators
// PBU (PB if the graph does not contain any non-trivial strong components) 
// and Hybrid-Lookahead.
//
// If the graph contains non-trivial strong components the D-Operator (SelectSupported)
// is used for Choice-Point selection otherwise the C-Operator (SelectUncolored)
// is used. In either way the Hybrid-Lookahead-Heuristic is used. 
// 
// Choices that should be positive are colored to the node's preferred choice-point color
// (see Node::preferredChoicePointColor). Choices that should be negative
// are colored to the node's alternative choice-point color 
// (see Node::alternativeChoicePointColor)
///////////////////////////////////////////////////////////////////////////////
class Expander::Impl
{
public:
    Impl(const ProgramInterface& program, const DelegatableChoice& dc);
    ~Impl();
    inline bool hasUnknown() const;
    inline bool hasConflict() const;
    PartialAssignment partialAssignment() const;
    inline size_t unknownAtoms() const;
    void expand(const Choice& c);
    void reinitialize(const DelegatableChoice& dc);
    void backtrackTo(const Choice& c);
    inline Choice getChoice() const;

    PossibleChoices possibleChoices() const;
private:
    Impl(const Impl&);
    Impl& operator=(const Impl&);

    typedef std::pair<Color::ColorValue, Color::ColorValue> ColorPair;

    void initGraph(const DelegatableChoice& dc);
    void expand();
    inline Node* getNode(const Choice& c);
    void setNextChoice();
    void createOperators();
    
    inline ColorPair getChoiceColors(Node* n, const Choice& c) const;
    
    const Program* program_;
    bool hasConflict_;
    Node* nextChoice_;
    std::auto_ptr<Graph> graph_;
    std::auto_ptr<Operator> pbuWithLookahead_;
    std::auto_ptr<ChoiceOperator> choiceOp_;

    typedef std::deque<Choice> ChoiceStack;
    ChoiceStack choiceStack_;
    
    
};

///////////////////////////////////////////////////////////////////////////////
// class NomoreProgram::Impl - public interface
///////////////////////////////////////////////////////////////////////////////
Expander::Impl::Impl(const ProgramInterface& p, const DelegatableChoice& dc)
    : program_(dynamic_cast<const Nomore::Program*>(&p))
    , hasConflict_(false)
    , nextChoice_(0)
{
    if (!program_) 
    {
        throw TypeError("Expander requires NomoreProgram");
    }
    initGraph(dc);
}

Expander::Impl::~Impl() 
{}

bool Expander::Impl::hasUnknown() const
{
    assert(graph_.get() && "Expander: graph missing!");
    return !graph_->totalColoring();
}

bool Expander::Impl::hasConflict() const 
{
    return hasConflict_;
}

PartialAssignment Expander::Impl::partialAssignment() const
{
    assert(graph_.get() && "Expander: graph missing!");
    PartialAssignment pa(*program_);
    for (Graph::HeadNodeIterator it = graph_->headNodesBegin(); it != graph_->headNodesEnd(); ++it)
    {
        if ( (*it)->getColor() == Color::plus )
        {
            pa.setTrue( Atom(static_cast<AtomId>((*it)->getId()), *program_) );
        }
        else if ( (*it)->getColor() == Color::minus )
        {
            pa.setFalse( Atom(static_cast<AtomId>((*it)->getId()), *program_) );
        }
    }
    return pa;
}

size_t Expander::Impl::unknownAtoms() const
{
    assert(graph_.get() && "Expander: graph missing!");
    size_t count = 0;
    for (Graph::HeadNodeIterator it = graph_->headNodesBegin(); it != graph_->headNodesEnd(); ++it) {
        if ( Color::isUncolored((*it)->getColor()) )
            ++count;
    }
    return count;
}
 
void Expander::Impl::expand(const Choice& c)
{
    assert(graph_.get() && "Expander: graph missing!");
    Node* n = getNode(c);
    if (n->getColor() != Color::none)
    {
        throw InvalidChoice();
    }
    try
    {
        ColorPair p = getChoiceColors(n, c);    
        graph_->colorChoicePoint(*n, p.first, p.second);
        choiceStack_.push_front(c);
        expand();
    }
    catch(const ColorError&)
    {
        hasConflict_ = true;
    }
}

void Expander::Impl::reinitialize(const DelegatableChoice& dc)
{
    delete pbuWithLookahead_.release();
    delete graph_.release();
    delete choiceOp_.release();
    choiceStack_.clear();
    nextChoice_ = 0;
    hasConflict_ = false;
    initGraph(dc);
}

void Expander::Impl::backtrackTo(const Choice& c)
{
    assert(graph_.get() && "Expander: graph missing!");
    
    // first restore choices made after c...
    while (!choiceStack_.empty() && choiceStack_.front().id() != c.id()) 
    {
    	bool debugBool = graph_->restoreChoicePoint();
        assert(debugBool && "stack empty but choice not yet reached!");
        choiceStack_.pop_front();
    }
    
    if (choiceStack_.empty()) 
    {	// ups - c is an unknown choice!
    	throw NoChoiceMade();
   	}
    
    // ...then restore c
    choiceStack_.pop_front();
    
    try
    {
        bool debugBool = graph_->recolorChoicePoint(*choiceOp_);
        assert(debugBool && "stack empty but choice not yet reached!");
        expand();
    }
    catch(const ColorError&)
    {
        hasConflict_ = true;
    }
}

Choice Expander::Impl::getChoice() const
{
    assert(graph_.get() && "Expander: graph missing!");
    if (nextChoice_)
    {
        return Choice(nextChoice_->getId(), true);
    }
    throw NoChoiceLeft();
}

PossibleChoices Expander::Impl::possibleChoices() const
{
    PossibleChoices pc;
    for (Graph::UncoloredNodeIterator it = graph_->uncoloredNodesBegin();
        it != graph_->uncoloredNodesEnd(); ++it)
    {
        pc.push_back((*it)->getId());
    }
    return pc;
}
///////////////////////////////////////////////////////////////////////////////
// class NomoreProgram::Impl - private helpers
///////////////////////////////////////////////////////////////////////////////
void Expander::Impl::initGraph(const DelegatableChoice& dc)
{
    graph_.reset(program_->getImplementation()->cloneGraph());
    hasConflict_ = program_->getImplementation()->hasConflict();
    createOperators();
    
    if (!hasConflict_)
    {
        try
        {
            // the delegatable choices are no real choices in the sense
            // that they are not backtrackable.
            for (DelegatableChoice::const_iterator it = dc.begin(); it != dc.end(); ++it)
		    {
			    Node* n = getNode(*it);
                ColorPair p = getChoiceColors(n, *it);
                graph_->color(*n, p.first);
		    }
            expand();
        }
        catch(const ColorError&)
        {
            hasConflict_ = true;
        }
    }
    graph_->finishConstruction();
}

void Expander::Impl::createOperators() {
    assert(graph_.get());
    
    bool useUOperator = program_->getImplementation()->needsUOperator();
    if (useUOperator) {
        choiceOp_.reset(new SelectSupported(*graph_, new HybridLookaheadHeuristic));
    }
    else {
        choiceOp_.reset(new SelectUncolored(*graph_, new HybridLookaheadHeuristic));
    }

    
    std::auto_ptr<Sequence> propOps(new Sequence(true));
    propOps->add(new ForwardPropagator(*graph_));
    propOps->add(new BackwardPropagator(*graph_));
    
    /* U-operator currently to slow!   
    if (useUOperator) {
             propOps->add(new UnfoundedSetOperator(*graph_, false));
    }
    */
    
    std::auto_ptr<Lookahead> lookaheadOp(new HybridLookahead(*graph_));
    lookaheadOp->setPropagationOperator(*propOps);
    lookaheadOp->setConstraint(choiceOp_->getConstraint());
    lookaheadOp->setHeuristic(choiceOp_->getHeuristic());

    pbuWithLookahead_.reset(new Sequence(propOps.release(), lookaheadOp.release()));

}

void Expander::Impl::expand()
{
    hasConflict_ = false;
    (*pbuWithLookahead_)();
    if (!graph_->totalColoring())
    {
        setNextChoice();
    }
}

Node* Expander::Impl::getNode(const Choice& c)
{
    return graph_->getNode(c.id());
}

void Expander::Impl::setNextChoice()
{
    nextChoice_ = choiceOp_->selectChoice();
    if (!nextChoice_)
    {
        bool debugBool = graph_->totalize();
        (void)debugBool;
        assert(debugBool && "Nomore-Expander: internal error");
    }
}

Expander::Impl::ColorPair Expander::Impl::getChoiceColors(Node* n, const Choice& c) const
{
    ColorPair p(n->preferredChoicePointColor(), n->alternativeChoicePointColor());
    if (!c.isPositive())
    {
        std::swap(p.first, p.second);
    }
    return p;
}


///////////////////////////////////////////////////////////////////////////////
// class Expander
// 
// 
///////////////////////////////////////////////////////////////////////////////
Expander::Expander(const ProgramInterface& program, const DelegatableChoice& dc)
    : impl_(new Impl(program, dc))
{
    createStrategy(this, dc);
}

Expander::~Expander()
{
    delete impl_;
}
    
bool Expander::hasUnknown() const
{
    return impl_->hasUnknown();
}

bool Expander::hasConflict() const
{
    return impl_->hasConflict();
}

PartialAssignment Expander::partialAssignment() const
{
    return impl_->partialAssignment();
}

size_t Expander::unknownAtoms() const
{
    return impl_->unknownAtoms();
}

void Expander::strategyExpand(const Choice& choice)
{
    return impl_->expand(choice);
}

void Expander::strategyReinitialize(const DelegatableChoice& dc)
{
    return impl_->reinitialize(dc);
}

void Expander::strategyBacktrackTo(const Choice& c)
{
    return impl_->backtrackTo(c);
}

Choice Expander::strategyGetChoice()
{
    return impl_->getChoice();
}

PossibleChoices Expander::possibleChoices() const
{
    return impl_->possibleChoices();
}

} }   // end namespace Platypus::Nomore

namespace Platypus {

///////////////////////////////////////////////////////////////////////////////
// factory registration code
///////////////////////////////////////////////////////////////////////////////
static Expander* create(const ProgramInterface& program, const DelegatableChoice& dc) 
{
    return new Nomore::Expander(program, dc);
}

template<>
void factoryRegistration<Platypus::Nomore::Expander>()
{
    ExpanderFactory::instance().add("nomore", &create);
}

}
