/*	$Id: options_builder.cpp 2360 2005-08-22 10:39:06Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <sstream>
#include <fstream>
#include <program_options/program_options.h>
#include <program_options/value.h>
#include <platypus/default_types.h>
#include <platypus/types_factory.h>
#include <platypus/exceptions.h>
#include <platypus/builder/options.h>
#include <platypus/builder/options_builder.h>

using namespace ProgramOptions;
using namespace std;

namespace Platypus
{
	std::string PlatypusOptionsBuilder::usage() const
	{
		// repolulate here to add stuff in modules
		// that was loaded at a later point
		populateOptionGroup();

		ostringstream s;
		s << *options_;
		return s.str();
	}

	PlatypusOptionsBuilder::~PlatypusOptionsBuilder()
	{}
	PlatypusOptionsBuilder::PlatypusOptionsBuilder()
	{
		populateOptionGroup();
	}
	std::string PlatypusOptionsBuilder::name() const
	{
		ostringstream s;
		s	<< "usage: platypus" 
			<< " [-e <answer set solver>]"
			<< " [-l <local choice heuristic>]"
			<< " [-d <delegatable choice heuristic>]"
			<< " [-f <filename>]"
			<< " [-a 0..n]"
			<< " [-s]"
			<< " [-c st|mt]"
			<< " [-t 1..n]"
			<< " [-m <distribution mode>]"
			<< "\n\n";

		
		s << "By default platypus reads from stdin.";

		return s.str();		
	}
	void PlatypusOptionsBuilder::populateOptionGroup() const
	{
		options_.reset(new OptionGroup(name()));

		ostringstream core, dcPolicy, localPolicy, mode, solver;

		CoreFactory::key_chain_type ckc = CoreFactory::instance().keys();
		CoreFactory::key_description_chain_type cdc = CoreFactory::instance().descriptions();
		core << "Core options:\n";
		for(size_t i = 0; i < ckc.size(); ++i)
		{
			core << "\t" << ckc[i] << ": " << cdc[i] << "\n";
		}		
		core << "\tDefault is '" << DEFAULT_CORE << "'\n";
		
		DCPolicyFactory::key_chain_type dkc = DCPolicyFactory::instance().keys();
		DCPolicyFactory::key_description_chain_type ddc = DCPolicyFactory::instance().descriptions();
		dcPolicy << "Policy to use for delegatable choices:\n";
		for(size_t i = 0; i < dkc.size(); ++i)
		{
			dcPolicy << "\t" << dkc[i] << ": " << ddc[i] << "\n";
		}
		dcPolicy << "\tBy default '" << DEFAULT_DELEGATABLE_CHOICE_POLICY << "' is used.\n\n";

		ChoicePolicyFactory::key_chain_type lkc = ChoicePolicyFactory::instance().keys();
		ChoicePolicyFactory::key_description_chain_type ldc = ChoicePolicyFactory::instance().descriptions();
		localPolicy << "Choice to pick for local propagation:\n";
		for(size_t i = 0; i < lkc.size(); ++i)
		{
			localPolicy << "\t" << lkc[i] << ": " << ldc[i] << "\n";
		}
		localPolicy << "\tBy default the 'expander' decides.\n";


		ExpanderFactory::key_chain_type ekc = ExpanderFactory::instance().keys();
		ExpanderFactory::key_description_chain_type edc = ExpanderFactory::instance().descriptions();
		solver << "Answer set solver to use:\n";
		for(size_t i = 0; i < ekc.size(); ++i)
		{
			solver << "\t" << ekc[i] << ": " << edc[i] << "\n";
		}
		solver << "\tBy default '" << DEFAULT_EXPANDER << "' is used.\n";

		DistributionFactory::key_chain_type mkc = DistributionFactory::instance().keys();
		DistributionFactory::key_description_chain_type mdc = DistributionFactory::instance().descriptions();
		mode << "Distribution mode:\n";
		for(size_t i = 0; i < mkc.size(); ++i)
		{
			mode << "\t" << mkc[i] << ": " << mdc[i] << "\n";
		}
		mode << "\tBy default no distribution is used.\n";

		options_->addOptions()
			("help,h", bool_switch()->defaultValue(false), "Print help and exit")			// <- kein Semikolon!
			
			("silent,s", bool_switch()->defaultValue(false), "Suppress printing of answer sets")
			("debug-prints", bool_switch()->defaultValue(false), "Print debugging information")
			("file,f",  value<string>(), "Read program from file")
			("answerSets,a", value<int>()->defaultValue(1), "Answer sets to calculate (0 = all)")
			("localChoicePolicy,l", value<string>()->defaultValue(DEFAULT_CHOICE_POLICY), localPolicy.str().c_str())
			("delegatableChoicePolicy,d", value<string>()->defaultValue(DEFAULT_DELEGATABLE_CHOICE_POLICY), dcPolicy.str().c_str())
			("threads,t", value<int>()->defaultValue(2), "Number of threads to use in multi-threaded core (2)")
			("delegationThreshold", ProgramOptions::value<int>()->defaultValue(10), "Percentage of atoms that must be UNKNOWN in order to delegate (10)")			
			("expander,e", value<string>()->defaultValue(DEFAULT_EXPANDER), solver.str().c_str())
			("core,c", value<string>()->defaultValue(DEFAULT_CORE), core.str().c_str())			
			("mode,m", value<string>()->defaultValue(DEFAULT_DISTRIBUTION), mode.str().c_str());

		addProbeOptions();
	}

	
	void PlatypusOptionsBuilder::addProbeOptions() const
	{
		ostringstream policy;

		ProbingPolicyFactory::key_chain_type pkc = ProbingPolicyFactory::instance().keys();
		ProbingPolicyFactory::key_description_chain_type pdc = ProbingPolicyFactory::instance().descriptions();
		policy << "Probing policy:\n";
		for(size_t i = 0; i < pkc.size(); ++i)
		{
			policy << "\t" << pkc[i] << ": " << pdc[i] << "\n";
		}
		policy << "\tBy default '" << DEFAULT_PROBING_POLICY << "' is used.\n";

		options_->addOptions()
			("probing", bool_switch()->defaultValue(false), "Enable probing.")
			("probing-initial-cutoff", value<int>()->defaultValue(100), "Initial # of backtracks before restart (100)")
			("probing-steps", value<int>()->defaultValue(64), "Steps after which # backtracks is doubled (64)")
			("probing-bestn", value<int>()->defaultValue(0), "Use best n elements to choose from (0 = all)")
			("probing-policy", value<string>()->defaultValue(DEFAULT_PROBING_POLICY), policy.str().c_str())
			;
	}
	void PlatypusOptionsBuilder::checkProbeValues()
	{
		try
		{
			const int v = option_as<int>(*values_, "probing-bestn");
			if(v < 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("Probing best N must be a non-negative number!");
		}
		try
		{
			const int v = option_as<int>(*values_, "probing-initial-cutoff");
			if(v <= 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("Initial probing cutoff must be positive!");
		}

		try
		{
			const int v = option_as<int>(*values_, "probing-steps");
			if(v <= 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("Probing steps must be positive!");
		}
	}
	PlatypusOptions* PlatypusOptionsBuilder::build(int& argc, char** argv)
	{
		values_.reset(new OptionValues);
		try
		{
			values_->store(ProgramOptions::parseCommandLine(argc, argv, *options_, true));
		}
		catch(const exception& e)
		{
			throw CommandlineError(e.what());
		}
		checkValues();
		auto_ptr<PlatypusOptions> po(new PlatypusOptions);
		po->options(*values_);
		return po.release();
	}

	void PlatypusOptionsBuilder::checkValues()
	{
		assert(values_.get());
		try
		{
			int noThreads = option_as<int>(*values_, "threads");
			if(noThreads <= 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("The number of threads to use must be greater than 0!");
		}

		try
		{
			int noAnswerSets = option_as<int>(*values_, "answerSets");
			if(noAnswerSets < 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("The number of answer sets to calculate must be non-negative!");
		}

		try
		{
			const int percentage = option_as<int>(*values_, "delegationThreshold");
			if(percentage < 0 || percentage > 100)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("Delegation-threshold  must be in the range [0..100]!");
		}

		if(values_->count("file"))
		{
			const string filename(option_as<string>(*values_, "file"));
			ifstream in(filename.c_str());
			if(!in)
				throw InputError("Could not open file " + filename + " for reading!");
		}

		checkProbeValues();
	}
}

