/////////////////////////////////////////////////////////////////////////////
//	file		:	filter_utils.h
//  copyright	:	(C) 2002-2004 Benjamin Kaufmann
//  email		:	hume@c-plusplus.de
//	internet	:	http://bens.c-plusplus.info/
//
//  Definition des Filter-Combiner-Interfaces
/////////////////////////////////////////////////////////////////////////////
#ifndef FILTER_UTILS_H_INCLUDED
#define FILTER_UTILS_H_INCLUDED
/*! \defgroup BoolFilter Expression-Filter und logische Operationen
\par Motivation
Hufig lsst sich ein Filterkriterium als Kombination einfacherer Filter
ausdrcken, z.B. knnte ein Anwender alle cpp-Dateien die nicht temporre
Dateien sind auswhlen wollen. 
Unter der Annahme, dass der Anwender bereits eine NoTemporaries-Filter-Klasse
implementiert hat, knnte er nun folgendes schreiben:
\code
class CppsNotTemporary
{
public:
    CppsNotTemporary()
		: cpp_f_("*.cpp")
	{}
	bool operator()(const std::string& e) const
	{
		return cpp_f_(e) && !notTemp_f_(e);
	}
private:
	pattern_f cpp_f_;
	NoTemporaries notTemp_f_;
};
...
dirstream str(".", pred_f(CppsNotTemporary()));
...
\endcode
Die Implementation solcher Funktionsobjekte ist zwar einfach, fhrt aber schnell
zu einer explosionsartigen Anhufung kleiner Prdikate, die sich nur schlecht
wiederverwenden lassen. Aus diesem Grund bietet dirstream die Mglichkeit
Filter ber logische Operationen zu verknpfen. Intern wird dann automatisch
ein \ref Filter "Filter-Objekt" erzeugt, das den fertigen Ausdruck reprsentiert.
\par Benutzung
Die Benutzung des Expression-Filter Frameworks ist denkbar einfach.
Anstelle eines einfachen Filters bergibt man einen boole'schen Ausdruck an
den Konstruktor von dirstream.
\code
dirstream str(".", op(pattern_f("*.cpp")) && !op(NoTemporaries()));
\endcode
Wie man dem Code entnehmen kann, mssen Filter, die ber logische Operationen
verknpft werden sollen, zuvor an die Funktion \ref dirstr::op "op" bergeben werden.
Dieser kleine Umweg ist ntig, damit auch Zeiger auf Filter bergeben werden
knnen (Ein berladener Operator muss mindestens ein Argument vom Typ UDT besitzen).
\code
filter_base* f1 = filterFactory.getFilterA();
filter_base* f2 = filterFactory.getFilterB();
dirstream str(".", op(f1) || op(f2));
delete f2;
delete f1;
\endcode
\par Arbeitsweise
Die Kombination von Filtern geschieht intern ber <a href="http://osl.iu.edu/~tveldhui/papers/Expression-Templates/exprtmpl.html">Expression-Templates</a>.
Das entstehende Expression-Objekt wird dann in ein \ref Filter "Filter-Objekt"
transformiert. Die Implementation des Expression-Frameworks ist relativ kompliziert 
(im Vergleich zur Komplexitt der restlichen Bibliothek), fr den Anwender zum Glck aber
auch irrelevant.\n
\b Anwender des Expression-Frameworks <b>mssen lediglich drei Dinge beachten</b>:
-# Klassen im Namespace dirstr::detail drfen nicht direkt verwendet werden. Sie
sind Implementationsdetails der Bibliothek.
-# Das Ergebnis eines boole'schen Ausdrucks darf nur in drei Kontexten verwendet werden
	-# als Argument fr die Funktion \ref dirstr::expr_f "expr_f"
	-# als Argument fr einen Filter-Parameter eines \ref dirstr::dirstream "dirstream-Konstruktors"
	-# als Argument fr einen Filter-Parameter einer \ref dirstr::dirstream "dirstream::open"-Funktion
-# Bei der Umwandlung eines Expression-Objekts in ein \ref Filter "Filter-Objekt" werden alle enthaltenen 
Filter kopiert. D.h konkret:
	- \ref Filter "Filter-Objekte", Prdikate und Funktionszeiger werden ber ihren Copy-Konstruktor kopiert.
	- Zeiger auf Objekte einer von \ref dirstr::filter_base "filter_base" abgeleiteten Klasse 
	werden ber den Aufruf der Methode \ref dirstr::filter_base::clone "clone" kopiert.
	- Der Value-Type eines SharedPtrs wird gemt der ersten beiden Regeln kopiert.
	.
\attention
Da Filter-Objekte und Prdikate ber ihren Copy-Konstruktor kopiert werden,
kann es hier unter Umstnden zu Slicing-Problemen kommen, nmlich genau
dann, wenn der Anwender ein Objekt einer abgeleiteten Klasse ber eine
Referenz auf eine konkreten Basisklasse an die Funktion \ref dirstr::op "op" bergibt.
\code
class FilterA {...};
class FilterB : public FilterA {};
FilterB b;
FilterA& a = b;
op(a) // Ups! Aus FilterB wird ein FilterA
\endcode
\par Einschrnkungen
- Die Untersttzung beliebiger Objekt-Pointer ohne die Verwendung von partieller Templatespezialisierung 
lsst sich nur ber einen nichttrivialen Umweg implementieren. 
Da dirstream aber mglichst auch auf Compilern, die keine partielle Templatespezialisierung
erlauben, bersetzbar sein soll und da der Aufwand der Implementation des Umwegs hher als
der Nutzen ist, werden vorerst nur filter-base-kompatible Objekt-Pointer untersttzt. 
- Die Menge der untersttzten Objekte sowie deren Kopiersemantik ist derzeit fest vorgegeben
und kann nicht verndert werden. Hier wre eine konfigurierbarer Traits-Ansatz wnschenswert.
Da die Implementation eines solchen Ansatzes ohne die Verwendung von partieller Templatespezialisierung
aber recht aufwendig ist, wird ein solcher Ansatz erst verfolgt, wenn 
sein Nutzen durch einen konkreten Bedarf bewiesen ist.
 */
#include "detail/filter_utils_impl.h"
namespace dirstr {
	/**
	 * \ingroup BoolFilter
	 * \brief Markiert einen Filter als Operand fr eine logische Operation.
	 *
	 * \param f Ein filter-kompatibles Objekt.
	 *
	 * \par 
	 * filter-kompatible ist:
	 * - ein Objekt einer Klasse die von \ref dirstr::filter_base "filter_base" erbt
	 * - ein Zeiger auf ein Objekt einer Klasse die von \ref dirstr::filter_base "filter_base" erbt.
	 * - ein Funktionsobjekt das mit einem std::string-Argument
	 * aufgerufen werden kann und ein bool-kompatiblen Rckgabewert hat.
	 * - ein Zeiger auf eine Funktion die mit einem std::string-Argument
	 * aufgerufen werden kann und ein bool-kompatiblen Rckgabewert hat.
	 * - ein SharedPtr-Objekt dessen Value-Type eine der vorherigen Bedingungen erfllt
	 *
	 * \attention
	 *	Argumente die an op bergeben werden, gehen \b nicht in den Besitz von op ber.
	 *	Das gilt insbesondere auch fr Objekte die per Pointer bergeben werden.
	 *	D.h. Code wie:
	 *  \code
	 *	op(new pattern_f("*.cpp"));
	 *	\endcode
	 *	lsst sich zwar problemlos kompilieren, fhrt zur Laufzeit aber zu
	 *	einem \b Speicherloch.
	 *
	 * \return Ein Objekt einer nicht spezifizierten Klasse das als Argument
	 * fr einen logische Operator verwendet werden kann.
	 *
	 **/
	template <class F>
	detail::expr<F> op(const F& f)
	{
		return detail::expr<F>(f);
	}

	/**
	 * \ingroup BoolFilter
	 * \brief UND-verknpft die Ausdrcke e1 und e2.
	 *
	 * Der resultierende Filter liefert fr einen Eintrag e true, 
	 * gdw: (e1(e) != 0) && (e2(e) != 0)
	 *
	 * \par Beispiel-Benutzung:
	 * \code
	 * bool filter1(const std::string&); 
	 * bool filter2(const std::string&);
	 * ...
	 * filter_base* f = expr_f( op(&filter1) && op(&filter2) );
	 * ...
	 * delete f;
	 * \endcode
	 **/
	template <class E1, class E2>	
	detail::expr<detail::expr_binary_v<E1, E2, detail::and_op> >
	operator && (const detail::expr<E1>& e1, const detail::expr<E2>& e2) 
	{	
		return detail::expr<detail::expr_binary_v<E1, E2, detail::and_op> >( 
			detail::expr_binary_v<E1, E2, detail::and_op>(e1.f_, e2.f_) 
		); 
	} 
	
	/**
	 * \ingroup BoolFilter
	 * \brief ODER-verknpft die Ausdrcke e1 und e2.
	 *
	 * Der resultierende Filter liefert fr einen Eintrag e true, 
	 * gdw: (e1(e) != 0) || (e2(e) != 0)
	 *
	 * \see operator && (const detail::expr<E1>& e1, const detail::expr<E2>& e2) 
	 **/
	template <class E1, class E2>
	detail::expr<detail::expr_binary_v<E1, E2, detail::or_op> >
	operator || (const detail::expr<E1>& e1, const detail::expr<E2>& e2)
	{
		return detail::expr<detail::expr_binary_v<E1, E2, detail::or_op> >(
			detail::expr_binary_v<E1, E2, detail::or_op>(e1.f_, e2.f_)
		);
	}	

	/**
	 * \ingroup BoolFilter
	 * \brief XOR-verknpft die Ausdrcke e1 und e2.
	 *
	 * Der resultierende Filter liefert fr einen Eintrag e true, 
	 * gdw: (e1(e) != 0) ^ (e2(e) != 0)
	 *
	 * \see operator && (const detail::expr<E1>& e1, const detail::expr<E2>& e2) 
	 **/
	template <class E1, class E2>
	detail::expr<detail::expr_binary_v<E1, E2, detail::xor_op> >
	operator ^ (const detail::expr<E1>& e1, const detail::expr<E2>& e2)
	{
		return detail::expr<detail::expr_binary_v<E1, E2, detail::xor_op> >(
			detail::expr_binary_v<E1, E2, detail::xor_op>(e1.f_, e2.f_)
		);
	}	

	/**
	 * \ingroup BoolFilter
	 * \brief Negiert den Filter der durch den Ausdruck e reprsentiert wird
	 *
	 * Der resultierende Filter liefert fr einen Eintrag entry true, 
	 * gdw: e(entry) == false
	 *
	 * \par Benutzung:
	 * \code
	 * bool filter1(const std::string&); 
	 * ...
	 * filter_base* f = expr_f( !op(&filter1) );
	 * delete f;
	 * \endcode
	 **/
	template <class E>
	detail::expr<detail::expr_not_v<E> >
	operator!(const detail::expr<E>& e)
	{
		return detail::expr<detail::expr_not_v<E> >(
			detail::expr_not_v<E>(e.f_)
		);
	}

	/**
	 * \ingroup BoolFilter
	 * \brief Transformiert einen boole'schen-Filterausdruck in ein filter_base-Objekt.
	 *
	 * \return Liefert einen \ref dirstr::filter_base "filter_base"-Pointer auf ein dynamisch alloziertes 
	 * Objekt einer nicht spezifizierten Klasse.
	 * 
	 * \note Der Aufrufer ist fr die Zerstrung des Objekts verantwortlich.
	 **/
	template <class T>
	filter_base* expr_f(const dirstr::detail::expr<T>& e)
	{
		return e.clone();
	}

}	// namespace dirstr

#endif

