/*  Copyright (c) June 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifndef TOKEN_ISTREAM_H
#define TOKEN_ISTREAM_H

#include <istream>
#include <string>
#include <vector>
#include <cassert>
#include <streambuf>

namespace toolbox
{
	template<class T, class Traits = std::char_traits<T> >
	class basic_token_streambuf : public std::basic_streambuf<T, Traits>
	{
		typedef std::basic_streambuf<T, Traits> base_type;
		typedef std::vector<T> buffer_type;
		typedef typename Traits::char_type* PT;
		typedef const typename Traits::char_type* CPT;
	public:
		explicit
		basic_token_streambuf(std::istream& stream, CPT separator = 0, size_t size = 0)
			:	stream_(&stream)
			,	separator_(separator, separator + size)
		{
			assert(!separator_.empty());
			// first, disable write support in basic_streambuf ...
			base_type::setp(0, 0);
			// to data until we scan for token
			base_type::setg(0, 0, 0);
		}
		virtual std::streamsize xsgetn(PT to, std::streamsize size)
		{
			const std::streamsize extracted = 
				std::min(static_cast<std::streamsize>(base_type::egptr() - base_type::gptr()), size);
			Traits::copy(to, base_type::gptr(), extracted);
			base_type::setg(&token_[0], &token_[0] + extracted, &token_[0] + token_.size());
			return extracted;
		}
		virtual typename base_type::pos_type seekoff(typename base_type::off_type offset,
												std::ios_base::seekdir dir,
												std::ios_base::openmode which = std::ios_base::in | std::ios_base::out)
		{
			if(which & std::ios_base::out)
				return base_type::seekoff(offset, dir, which);

			if(dir == std::ios_base::cur)
				offset += static_cast<typename base_type::off_type>(base_type::gptr() - base_type::eback());
			else if(dir == std::ios_base::end)
				offset = static_cast<typename base_type::off_type>(token_.size()) - offset;

			return seekpos(offset, which);
		}
		virtual typename base_type::pos_type seekpos(typename base_type::pos_type offset,
												std::ios_base::openmode which = std::ios_base::in | std::ios_base::out)
		{
			if(which & std::ios_base::out)
				return base_type::seekpos(offset, which);

			if(offset < 0 || static_cast<size_t>(offset) > token_.size())
				offset = base_type::seekpos(offset, which);
			else
				// do seeking now that the offset has been transformed into 
				// a positive value counting from the beginning of the
				// buffer.
				base_type::setg(&token_[0], &token_[0] + offset, &token_[0] + token_.size());

			return static_cast<typename base_type::pos_type>(offset);
		}
		bool scan()
		{
			token_.clear();
			const bool good = stream_->good();
			if(good)
			{
				PT const BEGIN = &separator_[0];
				PT const END = BEGIN + separator_.size();
				PT t = BEGIN;
				while(stream_->good())
				{
					const int i = stream_->get();
					if(Traits::eq_int_type(i, Traits::eof()))
						break;

					const T c = Traits::to_char_type(i);
					if(Traits::eq(c, *t))
						++t;
					else
						t = BEGIN;
					
					token_.push_back(c);

					if(t == END)
					{
						// take out the token delimiter just read
						token_.resize(token_.size() - separator_.size());
						break;
					}
				}
			}
			base_type::setg(&token_[0], &token_[0], &token_[0] + token_.size());
			return good;
		}
		inline void stream(std::istream& stream)
		{
			stream_ = &stream;
		}
		inline std::istream& stream()
		{
			return *stream_;
		}
		inline void separator(CPT s, size_t size)
		{
			separator_.assign(s, s + size);
			assert(!separator_.empty());
		}
	private:
		basic_token_streambuf();
		basic_token_streambuf(const basic_token_streambuf&);
		basic_token_streambuf& operator=(const basic_token_streambuf&);
	private:
		std::istream* stream_;
		buffer_type token_, separator_;
	};

	typedef basic_token_streambuf<char> token_streambuf;
	typedef basic_token_streambuf<wchar_t> wtoken_streambuf;

	template<class T, class Traits = std::char_traits<T> >
	class basic_token_istream : public std::basic_istream<T, Traits> 
	{
		typedef std::basic_istream<T, Traits> base_type;
		typedef basic_token_streambuf<T, Traits> stream_buf_type;
		typedef std::basic_string<T, Traits> string_type;
		typedef typename Traits::char_type* PT;
		typedef const typename Traits::char_type* CPT;
	public:
		explicit
		basic_token_istream(base_type& stream, const string_type& s)
			:	base_type(0)
			,	buffer_(stream, s.c_str(), s.size())
		{
			assert(!s.empty());
			base_type::rdbuf(&buffer_);
		}
		explicit
		basic_token_istream(base_type& stream, CPT s)
			:	base_type(0)
			,	buffer_(stream, s, Traits::length(s))
		{
			assert(s);
			assert(Traits::length(s));
			base_type::rdbuf(&buffer_);
		}
		explicit
		basic_token_istream(base_type& stream, CPT s, size_t l)
			:	base_type(0)
			,	buffer_(stream, s, l)
		{
			assert(s);
			assert(l);
			base_type::rdbuf(&buffer_);
		}
		explicit
		basic_token_istream(base_type& stream, T c)
			:	base_type(0)
			,	buffer_(stream, &c, 1)
		{
			base_type::rdbuf(&buffer_);
		}
		inline bool scan()
		{
			return buffer_.scan();
		}
		inline void stream(std::istream& stream)
		{
			buffer_.stream(stream);
		}
		inline std::istream& stream()
		{
			return buffer_.stream();
		}
		inline void separator(const string_type& s)
		{
			assert(!s.empty());
			buffer_.separator(s.c_str(), s.size());
		}
		inline void separator(CPT s)
		{
			assert(s);
			assert(Traits::length(s));
			buffer_.separator(s, Traits::length(s));
		}
		inline void separator(CPT s, size_t l)
		{
			assert(s);
			assert(l);
			buffer_.separator(s, l);
		}
	private:
		basic_token_istream();
		basic_token_istream(const basic_token_istream&);
		basic_token_istream& operator=(const basic_token_istream&);
	private:
		stream_buf_type buffer_;
	};

	typedef basic_token_istream<char> token_istream;
	typedef basic_token_istream<wchar_t> wtoken_istream;
}


#endif

