/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_LOOKAHEAD_OPERATOR_H
#define NOMORE_LOOKAHEAD_OPERATOR_H

#include <operator.h>
#include <color.h>

namespace NS_NOMORE {

class BodyNode;
class HeadNode;
class Node;
class Constraint;
class Heuristic;

//! This abstract class is the base class for all lookahead operators.
/*!
 * Lookahead operators works in a conflict-driven way. They assign a color to
 * an uncolored node whenever the opposite assignment led to a conflict.
 * Conflic-detection is based on propagation using the set propagation-operator.
 *
 */
class Lookahead : public GraphOperator {
public:
  //! Creates the lookahead operator.
  /*! \param grp The graph the operator should work on. 
   * 
   */ 
  Lookahead(const std::string& name, Graph& grp);

  //! Creates the lookahead operator.
  /*! \param grp The graph the operator should work on. 
   *  \param op The operator used for propagtion. 
   *  \param con Constraint for superseding of nodes.
   *
   *  \note this constructor only stores a reference to the operator op
   *  but does not take over ownership of the operator. 
   */ 
  Lookahead(const std::string& name, Graph& grp, Operator& op, const Constraint& con);

  //! Creates the lookahead operator.
  /*! \param grp The graph the operator should work on. 
   *  \param op The operator used for propagtion. 
   *  \param con Constraint for superseding of nodes.
   *
   *  \note this constructor take over ownership of the operator op
   *  I.e. when this lookahead-operator is destroyed it will automatically
   *  destroy the operator op.
   */ 
  Lookahead(const std::string& name, Graph& grp, Operator* op, const Constraint& con);

  ~Lookahead();

  //! sets the propagation operator to be used by this lookahead operator
  /*!  
   * \note takes over ownership of the operator op
   *  I.e. when this lookahead-operator is destroyed it will automatically
   *  destroy the operator op.
   */ 
  void setPropagationOperator(Operator* op);
  
  //! sets the propagation operator to be used by this lookahead operator
  /*!  
   *  \note only stores a reference to the operator op
   *  but does not take over ownership of the operator. 
   */ 
  void setPropagationOperator(Operator& op);

  void setPropagationOperator(Operator* op, bool takeOverOwnership);

  //! Sets the constraint for test of supersedes of nodes.
  void setConstraint(const Constraint& con) {
    constraint_ = &con;
  }

  //! Gets the constraint for test of supersedes of nodes.
  const Constraint& getConstraint() const  {
    return *constraint_;
  }
  
  //! Sets the heuristics.
  virtual void setHeuristic(Heuristic& heu) = 0;

  const Operator& getPropagationOperator() const;
  Operator& getPropagationOperator();
protected:
  //! checks whether the given color would be a valid color for the given node
  /*!
   * This method first colors the given node to the given color and then 
   * propagates the new color using the stored operator. 
   * If the propagation is successful (i.e. the color is valid for the node)
   * the method returns the result of the execution of the deterministc operator.
   * Otherwise it simply propagates the thrown ColorError exception.
   *  \param node The node to check
   *  \param color The color to which the node should be colored.
   *  \return The result of the deterministic operator is returned.
   *  \post Nothing has been changed on the graph after the call to this method.
   *  \throw ColorError The exception is thrown if the node can't be colored
   *  to the given color. 
   */
  bool checkColor(Node &node, Color::ColorValue color);
  
  //! colors the given node to the given color and propagates the new color using the stored operator.
  bool color(Node &node, Color::ColorValue color);

  /*!
   * disables automatic update of the graph's uncolored node list 
   * and signals the start of this lookahead-operation by firing a
   * lookahead execution event with the state-member set to ActivityState::started.
   * 
   * \note concrete lookahead operators that work on uncolored nodes 
   * must always call Lookahead::execute before they start their own execution.
   *
   */
  virtual bool execute();

  /*!
   * reenables automatic update of the graph's uncolored node list and
   * signals the end of this lookahead-operation by firing a
   * lookahead execution event with the state-member set to ActivityState::finished.
   *
   * \note concrete lookahead operators that work on uncolored nodes 
   * must always call Lookahead::postExecute after they completed their own execution.
   */
  virtual void postExecute();   
private:
  Operator* operator_;
  bool ownsOperator_;
  const Constraint* constraint_;
};

//! Defines the possible states of an activity for example the LookaheadExecution event.
struct ActivityState {

  //! Values of possible state.
  enum Value {
    //! the activity is started.
    started = 1,
    //! the activity is finished.
    finished = 2,
  };
};

//! Event which is fired if the lookahead is started or finished.
struct LookaheadExecution {
  explicit LookaheadExecution(ActivityState::Value state) : state_(state) {
	}
	ActivityState::Value state_;
};

}

#endif
