/*	$Id: kernel.cpp 3090 2006-01-10 17:07:17Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <iostream>
#include <nop_stream.h>
#include <platypus/types.h>
#include <platypus/cores/kernel.h>

using namespace std;

namespace Platypus
{
	Kernel::~Kernel()
	{}
	Kernel::Kernel(KernelCallback* callback)
		:	choicesToDo_(0)
		,	callback_(callback)
		,	os_(0)
		,	suppressAnswerSetPrinting_(false)
	{}
	void Kernel::setCallback(KernelCallback& callback)
	{
		callback_ = &callback;
	}
	
	void Kernel::configure()
	{
		assert(callback_);
		suppressAnswerSetPrinting_ = callback_->silent();
		probingPolicy_.reset(callback_->probingPolicy());
		assert(probingPolicy_.get());
		probingPolicy_->setup(space_);
		probing_.reset(callback_->probingManager());
		assert(probing_.get());
		probing_->setSearchSpace(space_);
		probing_->setPolicy(*probingPolicy_);
		expander_.reset(callback_->expander());
		assert(expander_.get());
		atoms_ = callback_->atoms();
		os_ = callback_->output();
		assert(os_);
		dcPolicy_.reset(callback_->delegatableChoicePolicy());
		assert(dcPolicy_.get());
		dcPolicy_->setup(space_);
		threadDelegationPolicy_.reset(callback_->threadDelegationPolicy());
		assert(threadDelegationPolicy_.get());
		threadDelegationPolicy_->setup(space_);
		
		(*os_)	<< 'P' << callback_->processId() << 'T' << callback_->threadId() 
				<< " kernel configured: #atoms = " << atoms_ << ".\n";
		
	}
	void Kernel::initialize()
	{
		(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " reinitializing expander\n";
		expander_->reinitialize(choicesToDo_->core());
		callback_->markExpanderInitialization();
	}
	void Kernel::run(const Branch& dc)
	{
		space_.integrate(dc);
		assert(!space_.empty());

		while(!callback_->shutdown() && !probing_->exhausted())
		{
			ChoicesToDo ctd(space_, probing_->select()); // don't remove stack object is important b/c it closes the search space branch in it's dtor
			choicesToDo_ = &ctd;
			initialize();
			
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " entering iteration cycle\n";
			bool not_done = true;
			while(!callback_->shutdown() && not_done)
			{
				not_done = iteration();
			}
			choicesToDo_ = 0;
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " leaving iteration cycle\n";
		}
	}
	bool Kernel::iteration()
	{
		if(expander_->hasConflict())
		{
			return conflict();
		}
		else if (!expander_->hasUnknown())
		{
			return answerSet();
		}
		return expand();
	}
	bool Kernel::answerSet()
	{
		(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " found answer set\n";
		assert(!expander_->hasConflict());
		assert(!expander_->hasUnknown());

		callback_->markAnswerSet();
		probing_->markAnswerSet();
		if(!suppressAnswerSetPrinting_)
		{
			callback_->answerSet(expander_->partialAssignment());
		}
		
		if(choicesToDo_->hasChoice())
		{
			return backtrack();
		}
		return false;
	}
	bool Kernel::conflict()
	{
		assert(expander_->hasConflict());
		callback_->markConflict();
		if(choicesToDo_->hasChoice())
		{
			return backtrack();
		}
		return false;
	}
	
	bool Kernel::expand()
	{
		assert(expander_->hasUnknown());
		assert(!expander_->hasConflict());
		

		const Choice c(expander_->getChoice());
		choicesToDo_->add(c);

		if(callback_->delegate())
		{
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " attempt to delgate part of search space to other process\n";
			delegate();
		}
		if(callback_->branch())
		{
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " attempt to delgate part of search space to other thread\n";
			branch();
		}

		(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " expanding with " << c.id() << "\n";
		expander_->expand(c);
		return true;
	}
	void Kernel::delegate()
	{
		const DelegatableChoice dc(dcPolicy_->choose());
		if(!dc.empty())
		{
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << ' ' << "giving " << dc << " to other process\n";
			callback_->delegate(dc);
		}
	}
	void Kernel::branch()
	{
		DelegatableChoice dc(threadDelegationPolicy_->choose());
		if(!dc.empty())
		{
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << ' ' << "giving " << dc << " to other thread\n"; 
			callback_->branch(dc);
		}
	}
	bool Kernel::backtrack()
	{
		assert(choicesToDo_->hasChoice());
		do
		{
			(*os_) << 'P' << callback_->processId() << 'T' << callback_->threadId() << " backtracking expander\n";
			probing_->markBacktrack();				
			callback_->markBacktrack();

			const Choice choice = choicesToDo_->nextChoice();			
			expander_->backtrackTo(choice);
		}
		while(expander_->hasConflict() && choicesToDo_->hasChoice());

		if(probing_->restart() && expander_->hasUnknown() && !expander_->hasConflict())
		{
			choicesToDo_->release();
			return false;
		}
		return !expander_->hasConflict();
	}
}
