/*	$Id: types.h 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef DLL_FACTORY_H
#define DLL_FACTORY_H

#include <cassert>
#include <string>
#include <cstdio>
#include <cstdlib>
#include <map>
#include <vector>
#include <smartpointer.h>
#include <foreach.h>
#include <xplib/xplib.h>
#include <platypus/exceptions.h>
#include <platypus/library.h>

namespace Platypus
{
	class DLLLoader
	{
	public:
		typedef toolbox::StackPtr<XPDLLLinker> Linker;
		typedef std::vector<Linker> Linkers;
		typedef Linkers::const_iterator const_iterator;
		
		static DLLLoader& instance();
		~DLLLoader();
		void loadAllInDirectory(const std::string& dir, bool report_failure);
		const_iterator begin();
		const_iterator end();
	private:
		DLLLoader();
		static XPDLLLinker* load(const std::string& path, bool complete);
		static bool hasProperFunctions(XPDLLLinker& linker);
		static void initializeLibrary(XPDLLLinker& linker);
	private:
		Linkers linkers_;
	};

	

	template<class Traits>
	class DLLFactory
	{
	public:
		typedef typename Traits::return_type return_type;
		typedef typename Traits::function_type function_type;
		typedef std::vector<std::string> key_chain_type;
		typedef std::vector<std::string> key_description_chain_type;
	private:
		typedef DLLLoader::Linker Linker;
		typedef std::map<std::string, std::pair<function_type, std::string> > Functions;
		typedef typename Functions::value_type value_type;
		typedef typename Functions::iterator iterator;
		typedef typename Functions::const_iterator const_iterator;
	public:
		static DLLFactory& instance();
		// reports a collection of keys (strings) registered with the factory
		// at this moment
		key_chain_type keys() const
		{
			key_chain_type k;
			k.reserve(functions_.size());
			FOREACH(const value_type& vt, functions_)
			{
				k.push_back(vt.first);
			}
			return k;
		}
		// returns a collection of strings that correspond
		// to the collection of keys
		key_description_chain_type descriptions() const
		{
			key_description_chain_type d;
			d.reserve(functions_.size());
			FOREACH(const value_type& vt, functions_)
			{
				d.push_back(vt.second.second);
			}
			return d;
		}
		// add a type to the library
		// register under id with create function f and an optional description
		bool add(const std::string& id, function_type f, const std::string& desc = std::string())
		{
			return functions_.insert(std::make_pair(id, std::make_pair(f, desc))).second;
		}
		void remove(const std::string& id)
		{
			const iterator res = functions_.find(id);
			if(res != functions_.end())
				functions_.erase(res);
		}
		inline void name(const std::string& name)
		{
			name_ = name;
		}
		
		// various create functions take 0..2 parameters
		return_type create(const std::string& id) const
		{
			const_iterator res = functions_.find(id);
			if(res == functions_.end())
				throw TypeError(name_ + ": unknown type '" + id + "'");
			return res->second.first();
		}
		template<class A>
		return_type create(const std::string& id, const A& a) const
		{
			const_iterator res = functions_.find(id);
			if(res == functions_.end())
				throw TypeError(name_ + ": unknown type '" + id + "'");
			return res->second.first(a);
		}
		template<class A, class B>
		return_type create(const std::string& id, const A& a, const B& b) const
		{
			const_iterator res = functions_.find(id);
			if(res == functions_.end())
				throw TypeError(name_ + ": unknown type '" + id + "'");
			return res->second.first(a, b);
		}
	private:
		DLLFactory()
			:	name_(Traits::type())
		{
			//std::printf("%s\n", Traits::type());
			std::atexit(destroy);
		}
		DLLFactory(const DLLFactory&);
		DLLFactory& operator=(const DLLFactory&);
		static void destroy();
	private:
		std::string name_;
		Functions functions_;
		
		static DLLFactory* instance_;
	};

	// We are not using the Meyer singleton here 
	/*
	static Foo::instance()
	{
		static Foo instance_;
		return instance_;
	}
	*/
	// because VC 7 allows for more than one instance
	// of the singleton to be created. We cannot use 
	// an auto_ptr object either to take care of deleting
	// the heap-allocated singleton object hence we
	// attempt to register an explicit destroy function
	// with C/C++ standard function atexit

	template<class Traits>
	void DLLFactory<Traits>::destroy()
	{
		delete instance_;
	}

	template<class Traits>
	DLLFactory<Traits>& DLLFactory<Traits>::instance()
	{
		if(!instance_)
			instance_ = new DLLFactory;
		return *instance_;
	}

	template<class Traits>
	DLLFactory<Traits>* DLLFactory<Traits>::instance_ = 0;

}

#endif
