/*	$Id: smodels_expander.cpp 2141 2005-07-04 07:43:57Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <new> // placement new
#include <cassert>
#include <string>
#include <algorithm>
#include <cstdio>
#include <platypus/smodels_names.h>
#include <platypus/program_base.h>
#include <platypus/factories/expander_factory.h>
#include <platypus/factories/registration.h>
#include <platypus/types/atom.h>
#include <platypus/types/exceptions.h>
#include <platypus/types/expander_strategies.h>
#include <platypus/smodels/smodels_expander.h>
#include <platypus/smodels/smodels_program.h>
#include <platypus/smodels/smodels_expander_impl.h>
#include <smodels/smodels.h>

using namespace std;

namespace Platypus
{
	///////////////////////////////////////////////////////////
	// SmodelsExpander
	///////////////////////////////////////////////////////////
	typedef std::vector< ::Atom* > SmodelsDictionary;
	
	void SmodelsExpander::destroyDictionary()
	{
		assert(rawDictionary_.get());
		// destroy dictionary object 
		reinterpret_cast<SmodelsDictionary*>(rawDictionary_.get())->~SmodelsDictionary();
	}
	SmodelsExpander::~SmodelsExpander()
	{
		if(pimpl_)
			pimpl_->~SmodelsExpanderImpl();

		// destroy dictionary after a potentional SmodelsExpanderImpl object
		destroyDictionary();
	}
	SmodelsExpander::SmodelsExpander(const ProgramInterface& program, const DelegatableChoice& dc)
		:	program_(dynamic_cast<const SmodelsEnhancedProgram*>(&program))
		,	pimpl_(0)
		,	rawExpander_(new char[sizeof(SmodelsExpanderImpl)])
		,	rawDictionary_(new char[sizeof(SmodelsDictionary)])
	{
		if(!program_)
			throw TypeError("Passed non SmodelsEnhancedProgram to SmodelsExpander c'tor!");

		createStrategy(this, dc);

		// create dictionary object with the necessary space
		new (rawDictionary_.get()) SmodelsDictionary(program_->numberOfAtoms()+1, 0);
		try
		{
			// attempt to create an expander impl object, if
			// this fails we need to destroy the Dictionary object manually
			reinitialize(dc);
		}
		catch(...)
		{
			destroyDictionary();
			throw;
		}
	}

	bool SmodelsExpander::hasUnknown() const
	{
		return pimpl_->hasUnknown();
	}
	bool SmodelsExpander::hasConflict() const
	{
		return pimpl_->hasConflict();
	}
	void SmodelsExpander::strategyExpand(const Choice& choice)
	{
		pimpl_->makeChoice(choice.id(), choice.isPositive());
	}
	void SmodelsExpander::strategyReinitialize(const DelegatableChoice& dc)
	{
		if(pimpl_)
		{
			SmodelsExpanderImpl* temp = pimpl_;
			pimpl_ = 0;
			temp->~SmodelsExpanderImpl();
		}
		pimpl_ = new (rawExpander_.get()) SmodelsExpanderImpl(*program_, dc, *reinterpret_cast<SmodelsDictionary*>(rawDictionary_.get()));
	}
	void SmodelsExpander::strategyBacktrackTo(const Choice& choice)
	{
		pimpl_->backtrackTo(choice.id());
	}
	Choice SmodelsExpander::strategyGetChoice()
	{
		return pimpl_->getChoice();
	}
	PossibleChoices SmodelsExpander::possibleChoices() const
	{
		return pimpl_->possibleChoices();
	}

	PartialAssignment SmodelsExpander::partialAssignment() const
	{
		return pimpl_->partialAssignment();
	}

	size_t SmodelsExpander::unknownAtoms() const
	{
		return pimpl_->unknownAtoms();
	}

	Expander* SmodelsExpander::create(const ProgramInterface& program, const DelegatableChoice& dc)
	{
		return new SmodelsExpander(program, dc);
	}

	template<>
	void factoryRegistration<SmodelsExpander>()
	{
		// cannot use constants here like SMODELS_EXPANDER_KEY b/c this registration
		// is done statically and unfortunately there is no order on static initialization.
		// This means that the const char* const might not have been properly initialized yet when 
		// this function is called. This appears to be a bug in the VC 7 compiler, 
		// according to the Standard constants should have been initialized at this
		// point.
		// This doesn't even work in debug mode after a fresh build.
		ExpanderFactory::instance().add("smodels", SmodelsExpander::create, "Use smodels");
	}
}
