/*  Copyright (c) October 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef FREE_HIGH_BITS_MUXER_H
#define FREE_HIGH_BITS_MUXER_H

#ifndef PT_ARCH_COMMON_INCLUDED
#	error "Architectural types and functions must be included first!"
#endif 

#include <cassert>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			// The assumption is a pointer is POINTER_BITS bits wide. The
			// pointer doesn't use all bits to represent a datum in memory
			// but only HARDWARE_POINTER_BITS bits. The other bits are 
			// either all zero or all ones:
			// [0...0|pointer] or [1...1|pointer]
			//
			// The muxing code multiplexes a (POINTER_BITS-HARDWARE_POINTER_BITS-1)
			// bit counter into the highest bits of the pointer while preserving
			// the most significant bit to indicate whether the original pointer
			// had its high bits set to all ones or all zeros.
			//
			// We assume the least ALIGNMENT_BITS of the pointer are zero because data
			// is aligned in memory.
			template<typename T, unsigned POINTER_BITS, unsigned HARDWARE_POINTER_BITS, unsigned ALIGNMENT_BITS = 0>
			struct FreeHighBits
			{
				typedef T int_t;
				static inline int_t multiplex(int_t pointer, int_t count)
				{
					// create: [half-indicator bit|pointer|count]
					assert(HARDWARE_POINTER_BITS < POINTER_BITS);
					assert(ALIGNMENT_BITS < HARDWARE_POINTER_BITS);

					// Make sure the aligment assumption holds
					assert((pointer & ALIGNMENT_MASK) == 0);

					// counter must fit in reserved bits, we save one bit to indicate 
					// whether pointer had high ones or high zeros
					count %= COUNTER_MAX;

					// move pointer to high region, preserve msb
					pointer <<= POINTER_SHIFT;
					assert((pointer & COUNTER_MASK) == 0);

					return pointer ^ count;
				}
				static inline int_t count(int_t mux)
				{
					return mux & COUNTER_MASK;
				}
				static inline int_t value(int_t mux)
				{
					// [half-indicator bit|pointer|count] -> [1..1|pointer] or [0..0|pointer]
					return  (mux & MSB_MASK) == 0 
							? (mux >> POINTER_SHIFT) & ~ALIGNMENT_MASK
							// has to be a true or (|) because the half-indicator bit is still there!
							: ((mux >> POINTER_SHIFT) & ~ALIGNMENT_MASK) | HIGH_ONES;
				}
				static inline int_t freeBits()
				{
					return POINTER_BITS - HARDWARE_POINTER_BITS;
				}
				static inline int_t pointerBits()
				{
					return POINTER_BITS;
				}
				static inline int_t hardwarePointerBits()
				{
					return HARDWARE_POINTER_BITS;
				}
				static inline int_t alignmentBits()
				{
					return ALIGNMENT_BITS;
				}
			private:
				static const int_t MSB_MASK = static_cast<int_t>(1) << (POINTER_BITS - 1);
				static const int_t COUNTER_MAX = static_cast<int_t>(1) << (POINTER_BITS - HARDWARE_POINTER_BITS + ALIGNMENT_BITS - 1);
				static const int_t COUNTER_MASK = COUNTER_MAX - 1;
				static const int_t ALIGNMENT_MASK = ((static_cast<int_t>(1) << ALIGNMENT_BITS) - 1);
				static const int_t POINTER_SHIFT = POINTER_BITS - HARDWARE_POINTER_BITS - 1;
				static const int_t HIGH_ONES = ((static_cast<int_t>(1) << (POINTER_BITS - HARDWARE_POINTER_BITS + 1)) - 1) << HARDWARE_POINTER_BITS;
			};
		}
	}
}


#endif
