/*  Copyright (c) March 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef MANUAL_SPINLOCK_POINTER_CAS_H
#define MANUAL_SPINLOCK_POINTER_CAS_H

#ifndef PT_ARCH_COMMON_INCLUDED
#	error "Architectural types and functions must be included first!"
#endif 

#define PT_POINTER_CAS_INCLUDED

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			class PTPointerCAS
			{
			public:
				typedef PTSizeToType_<sizeof(pt_pointer_type)>::int_type int_t;
				typedef PTSizeToType_<2*sizeof(int_t)>::int_type large_int_t;
			public:
				explicit
				PTPointerCAS(int_t v = 0)
					:	value_(v)
					,	lock_(0)
				{
					assign(v);
				}
				inline bool cas(int_t nv, const token_t& ov)
				{
					bool ret = false;
					const large_int_t ov_mplx = multiplex(ov.value(), ov.count());
					if(pt_atomic_set_lock(&lock_))
					{
						if(value_ == ov_mplx)
						{
							value_ = multiplex(nv, ov.count()+1);
							ret = true;
						}
						pt_atomic_clear_lock(&lock_);
					}
					return ret;
				}
				inline bool cas(token_t& nv, const token_t& ov)
				{
					bool ret = false;
					const large_int_t ov_mplx = multiplex(ov.value(), ov.count());
					if(pt_atomic_set_lock(&lock_))
					{
						if(value_ == ov_mplx)
						{
							value_ = multiplex(nv.value(), ov.count()+1);
							nv.count(ov.count()+1);
							ret = true;
						}
						pt_atomic_clear_lock(&lock_);
					}
					return ret;
				}
				inline token_t get() const
				{
					const large_int_t current = value_;
					return token_t(value(current), count(current));
				}
				inline void assign(int_t value)
				{
					value_ = multiplex(value, count(value_));
				}
			private:
				static inline int_t value(large_int_t large)
				{
					return static_cast<int_t>(large >> SHIFT);
				}
				static inline int_t count(large_int_t large)
				{
					return static_cast<int_t>(large);
				}
				static inline large_int_t multiplex(int_t value, int_t count)
				{
					return (static_cast<large_int_t>(ret) << SHIFT) ^ static_cast<large_int_t>(count);
				}
				PTPointerCAS(const PTPointerCAS&);
				PTPointerCAS operator=(const PTPointerCAS&);
			private:
				volatile large_int_t value_;
				volatile uint8 lock_;
			private:
				static const unsigned SHIFT = sizeof(int_t) * 8;
			};
		}
	}

}

#endif
