#include <cstdlib>
#include <algorithm>
#include <cassert>
#include <iostream>
#include <portablethreads/exception.h>
#include <portablethreads/lockfree/utility.h>
#include <portablethreads/lockfree/heap.h>

// in case windows.h is included
#ifndef max
#	undef max
#endif
#ifdef min
#	undef min
#endif

using namespace std;

namespace PortableThreads 
{

	namespace LockFree
	{

		/**************************************************************/
		/* PTHeapImpl                                                 */
		/**************************************************************/
		namespace Private
		{
			namespace
			{
				class MemoryChunk
				{
				public:
					MemoryChunk(void* raw)
						:	raw_(static_cast<char*>(raw))
					{}
					inline Private::token_t next() const { return next_.get(); }
					inline void next(Private::PTPointerCAS::int_t n) { next_.assign(n); }
					inline char* raw() const { return const_cast<char*>(raw_); }
				private:
					Private::PTPointerCAS next_;
					const char* raw_;
				};

				class Page
				{
				public:
					Page(void* raw)
						:	raw_(static_cast<char*>(raw))
					{}
					inline Private::token_t next() const { return next_.get(); }
					inline void next(Private::PTPointerCAS::int_t n) { next_.assign(n); }
					inline char* raw() const { return const_cast<char*>(raw_); }
				private:
					Private::PTPointerCAS next_;
					const char* raw_;
				};
			}
		
			class PTHeapImpl
			{
				typedef PTHeap::AllocateMemory AllocateMemory;
				typedef PTHeap::FreeMemory FreeMemory;
				typedef PTHeap::size_t size_t;
			public:
				~PTHeapImpl();
				explicit
				PTHeapImpl(AllocateMemory a, FreeMemory r);
				explicit
				PTHeapImpl(const PTHeapImpl&);
				PTHeapImpl& operator=(const PTHeapImpl&);
				void* allocate(size_t size);
				void deallocate(void* p);
				void swap(PTHeapImpl& other);
				static size_t smallestAllocatableBlock();
				static size_t largestAllocatableBlock();
			private:
				PTHeapImpl();
				static size_t findAlignFor(size_t where, size_t WHAT);
				static size_t getIndex(size_t s);
				void initializeStacks();
				void* allocatePage();
				void* heapAllocate(size_t size);
				void* cacheAllocate(size_t size);
				void addPage(size_t index);
				void addPage(char* page, size_t pagesize, size_t index);
			private:
				AllocateMemory allocate_;
				FreeMemory deallocate_;
				Private::PTPointerCAS* stacks_;
			private:
				static const size_t MIN_ALLOC_SIZE;
				static const size_t MAX_ALLOC_SIZE;
				static const size_t PAGE_SIZE;
				static const size_t ADJUST_INDEX;
				static const size_t MAX_INDEX;
				static const size_t PAGE_STACK_INDEX;
				static const size_t STRUCT_ALIGN;
				static const size_t SIZE_T_ALIGN;
				static const size_t ALIGNED_MEMORY_CHUNK_SIZE;
				static const size_t ALIGNED_SIZE_T_SIZE;
			};

			const size_t PTHeapImpl::MIN_ALLOC_SIZE = 64;
			const size_t PTHeapImpl::MAX_ALLOC_SIZE = 2048;
			const size_t PTHeapImpl::PAGE_SIZE = 4096;
			const size_t PTHeapImpl::ADJUST_INDEX = 6; // must satisfy 1 << ADJUST_INDEX == MIN_ALLOC_SIZE
			const size_t PTHeapImpl::MAX_INDEX = 11;  // must satisfy 1 << MAX_INDEX == MAX_ALLOC_SIZE
			const size_t PTHeapImpl::PAGE_STACK_INDEX = MAX_INDEX + 1 - ADJUST_INDEX; 
			const size_t PTHeapImpl::STRUCT_ALIGN = 2*sizeof(size_t);
			const size_t PTHeapImpl::SIZE_T_ALIGN = sizeof(size_t);
			const size_t PTHeapImpl::ALIGNED_MEMORY_CHUNK_SIZE = sizeof(MemoryChunk) + findAlignFor(sizeof(MemoryChunk), STRUCT_ALIGN);
			const size_t PTHeapImpl::ALIGNED_SIZE_T_SIZE = sizeof(size_t) + findAlignFor(sizeof(size_t), STRUCT_ALIGN);


			size_t PTHeapImpl::smallestAllocatableBlock()
			{
				return 64;
			}
			size_t PTHeapImpl::largestAllocatableBlock()
			{
				return 2048;
			}
			size_t PTHeapImpl::findAlignFor(size_t where, size_t WHAT)
			{
				assert(WHAT);
				assert(where);
				
				return WHAT - (where % WHAT);
			}

			size_t PTHeapImpl::getIndex(size_t s)
			{
				if(s <= MIN_ALLOC_SIZE)
					return ADJUST_INDEX;
				size_t index = ADJUST_INDEX;
				size_t cmp = static_cast<size_t>(1) << ADJUST_INDEX;
				while(cmp < s)
					++index, cmp <<= 1;

				return index;
			}


			PTHeapImpl::~PTHeapImpl()
			{
				
				const size_t s = getIndex(MAX_ALLOC_SIZE) - getIndex(MIN_ALLOC_SIZE) + 1;
				for(size_t i = 0; i < s; ++i)
				{
					for(MemoryChunk* m = 0; Private::stackPop(stacks_[i], m);)
					{
						m->~MemoryChunk();
					}
					(&stacks_[i])->~PTPointerCAS();
				}
				
				for(Page* p = 0; Private::stackPop(stacks_[PAGE_STACK_INDEX], p);)
				{
					assert(p);
					deallocate_(p->raw());
					p->~Page();
					deallocate_(p);
				}
				(&stacks_[PAGE_STACK_INDEX])->~PTPointerCAS();
				deallocate_(stacks_);
			}
			PTHeapImpl::PTHeapImpl(AllocateMemory a, FreeMemory r)
				:	allocate_(a)
				,	deallocate_(r)
				,	stacks_(0)
			{
				if(!allocate_)
				{
					throw PTParameterError("[PTHeap] Passed pointer to NULL for memory allocation function");
				}
				if(!deallocate_)
				{
					throw PTParameterError("[PTHeap] Passed pointer to NULL for memory deallocation function");
				}
				initializeStacks();
			}
			PTHeapImpl::PTHeapImpl(const PTHeapImpl& other)
				:	allocate_(other.allocate_)
				,	deallocate_(other.deallocate_)
				,	stacks_(0)
			{
				initializeStacks();
			}

			PTHeapImpl& PTHeapImpl::operator=(const PTHeapImpl& other)
			{
				PTHeapImpl(other).swap(*this);
				return *this;
			}
			void PTHeapImpl::initializeStacks()
			{
				assert(allocate_);
				assert(deallocate_);

				const size_t s = getIndex(MAX_ALLOC_SIZE) - getIndex(MIN_ALLOC_SIZE) + 2; // extra for page stack
				stacks_ = static_cast<PTPointerCAS*>(allocate_(sizeof(PTPointerCAS)*s));
				if(!stacks_)
					throw OutOfMemory();
				for(size_t i = 0; i < s; ++i)
				{
					new (&stacks_[i]) PTPointerCAS;
				}
			}
			void* PTHeapImpl::allocatePage()
			{
				Page* page = static_cast<Page*>(allocate_(sizeof(Page)));
				if(!page)
					throw OutOfMemory();
				try
				{
					// Page c'tor doesn't throw, only memory allocation may fail!
					void* p = allocate_(PAGE_SIZE);
					if(!p)
						throw OutOfMemory();
					new (page) Page(p);
				}
				catch(...)
				{
					deallocate_(page);
					throw;
				}

				Private::stackPush(stacks_[PAGE_STACK_INDEX], page);
				return page->raw();
			}
			void PTHeapImpl::swap(PTHeapImpl& other)
			{
				std::swap(allocate_, other.allocate_);
				std::swap(deallocate_, other.deallocate_);
				std::swap(stacks_, other.stacks_);
			}
			void* PTHeapImpl::allocate(size_t size)
			{
				if(size <= MAX_ALLOC_SIZE)
					return cacheAllocate(size);
				return heapAllocate(size);
			}
			void* PTHeapImpl::heapAllocate(size_t size)
			{
				assert(ALIGNED_SIZE_T_SIZE >= sizeof(size_t));

				char* p = static_cast<char*>(allocate_(size + ALIGNED_SIZE_T_SIZE));
				if(!p)
					throw OutOfMemory();
				*reinterpret_cast<size_t*>(p) = MAX_INDEX+1;
				
				return p + ALIGNED_SIZE_T_SIZE;
			}
			void PTHeapImpl::addPage(size_t index)
			{
				assert(sizeof(MemoryChunk) <= ALIGNED_MEMORY_CHUNK_SIZE);
				assert(sizeof(size_t) <= ALIGNED_SIZE_T_SIZE);

				char* p = static_cast<char*>(allocatePage());
				if(!p)
					throw OutOfMemory();

				addPage(p, PAGE_SIZE, index);
			}
			void PTHeapImpl::addPage(char* page, size_t pagesize, size_t index)
			{
				const size_t chunksize = static_cast<size_t>(1) << index;
				const size_t datasize = chunksize + ALIGNED_MEMORY_CHUNK_SIZE + ALIGNED_SIZE_T_SIZE;
				const size_t totalskip = datasize + findAlignFor(datasize, STRUCT_ALIGN);
				const size_t entries = pagesize / totalskip;
				
				for(size_t i = 0; i < entries; ++i, page += totalskip)
				{
					*reinterpret_cast<size_t*>(page + ALIGNED_MEMORY_CHUNK_SIZE) = index;
					MemoryChunk* m = new (page) MemoryChunk(page + ALIGNED_MEMORY_CHUNK_SIZE + ALIGNED_SIZE_T_SIZE);
					
					Private::stackPush(stacks_[index - ADJUST_INDEX], m);
				}

				if(index > ADJUST_INDEX)
					addPage(page, pagesize - entries * totalskip, index - 1);
			}
			void* PTHeapImpl::cacheAllocate(size_t size)
			{
				const size_t index = getIndex(size);
				assert(index >= ADJUST_INDEX);
				assert(index <= MAX_INDEX);

				MemoryChunk* b;		
				while(!Private::stackPop(stacks_[index - ADJUST_INDEX], b))
				{			
					addPage(index);
				}
				return b->raw();
			}

			void PTHeapImpl::deallocate(void* p)
			{
				assert(p);
				// find out if memory is managed by the cache
				char* c = static_cast<char*>(p) - ALIGNED_SIZE_T_SIZE;
				const size_t index = *reinterpret_cast<size_t*>(c);
				
				if(index <= MAX_INDEX)
				{
					assert(index >= ADJUST_INDEX);
					MemoryChunk* b = reinterpret_cast<MemoryChunk*>(c - ALIGNED_MEMORY_CHUNK_SIZE);
					Private::stackPush(stacks_[index - ADJUST_INDEX], b);
				}
				else
				{
					assert(index == MAX_INDEX+1);
					deallocate_(c);
				}	
			}
		}
	}

}

