/*  Copyright (c) June 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifndef RECYCLE_ALLOCATOR_H
#define RECYCLE_ALLOCATOR_H

#include <memory>
#include <vector>
#include <utility>
#include <cassert>

namespace toolbox
{
	template<class Allocator>
	class node_recycle_allocator
	{	
	public:
		typedef Allocator allocator_type;
		typedef typename allocator_type::value_type value_type;
		typedef typename allocator_type::reference reference;
		typedef typename allocator_type::const_reference const_reference;
		typedef typename allocator_type::pointer pointer;
		typedef typename allocator_type::const_pointer const_pointer;
		typedef typename allocator_type::difference_type difference_type;
		typedef typename allocator_type::size_type size_type;

		typedef std::vector<pointer, typename allocator_type::template rebind<pointer>::other> pool_type;
		typedef std::pair<pointer, size_type> memory_location_type;
		typedef std::vector<memory_location_type, typename allocator_type::template rebind<memory_location_type>::other> mem_pool_type;

		~node_recycle_allocator()
		{
			for(typename mem_pool_type::const_iterator it = memory_pool_.begin();
				it != memory_pool_.end(); ++it)
			{
				allocator_.deallocate(it->first, it->second);
			}
		}
		node_recycle_allocator()
			:	memory_pool_size_(inital_memory_pool_size_)
		{}
		node_recycle_allocator(const node_recycle_allocator& other)
			:	allocator_(other.allocator_)
			,	memory_pool_size_(inital_memory_pool_size_)
		{}
		template<class Other>
		node_recycle_allocator(const node_recycle_allocator<Other>& other)
			:	allocator_(other.allocator())
			,	memory_pool_size_(inital_memory_pool_size_)
		{}
		inline size_type max_size() const
		{
			return allocator_.max_size();
		}
		inline pointer address(reference t)
		{
			return allocator_.address(t);
		}
		inline const_pointer address(const_reference t) const
		{
			return allocator_.address(t);
		}
		inline pointer allocate(size_type count)
		{
			return allocate(count, static_cast<void*>(0));
		}
		template<class Other>
		pointer allocate(size_type count, const Other* hint = 0)
		{
			assert(count == 1);
			if(pool_.empty())
			{
				const size_type memsize = 1 << (memory_pool_size_++);
				memory_pool_.push_back(std::make_pair(allocator_.allocate(memsize, hint), memsize));

				pool_.reserve(memsize);

				pointer begin = memory_pool_.back().first;
				const pointer end = begin + memsize;
				while(begin != end)
					pool_.push_back(begin++);
			}

			pointer p = pool_.back();
			pool_.pop_back();
			return p;
		}
		inline void deallocate(pointer p, size_type count)
		{
			assert(count == 1);
			pool_.push_back(p);
		}
		inline void construct(pointer p, const_reference t)
		{
			allocator_.construct(p, t);
		}
		inline void destroy(pointer p)
		{
			allocator_.destroy(p);
		}
		template<class Other>
		struct rebind 
		{
			typedef node_recycle_allocator< typename allocator_type::template rebind<Other>::other > other;
		};
		
		inline const allocator_type& allocator() const { return allocator_; }
	
		void swap(node_recycle_allocator& other)
		{
			std::swap(allocator_, other.allocator_);
			std::swap(memory_pool_size_, other.memory_pool_size_);
			memory_pool_.swap(other.memory_pool_);
			pool_.swap(other.pool_);
		}
	private:
		node_recycle_allocator& operator=(const node_recycle_allocator&);
		template<class Other>
		inline node_recycle_allocator& operator=(const node_recycle_allocator<Other>& other)
		{
			allocator_ = other.allocator_;
			return *this;
		}	
	private:
		allocator_type allocator_;
		mem_pool_type memory_pool_;
		pool_type pool_;
		unsigned memory_pool_size_;

		static const unsigned inital_memory_pool_size_ = 3;
	};

	template<class T, class U>
	inline bool operator==(const node_recycle_allocator<T>& lhs, const node_recycle_allocator<U>& rhs)
	{
		return lhs.allocator() == rhs.allocator();
	}

	template<class T, class U>
	inline bool operator!=(const node_recycle_allocator<T>& lhs, const node_recycle_allocator<U>& rhs)
	{
		return !(lhs == rhs);
	}

}

// specialize std::swap
namespace std
{
	template<class T>
	inline void swap(toolbox::node_recycle_allocator<T>& lhs, toolbox::node_recycle_allocator<T>& rhs)
	{
		lhs.swap(rhs);
	}
}


#endif
