/*  Copyright (c) September 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SPARC_32_V8PLUS_GCC_H
#define SPARC_32_V8PLUS_GCC_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#define PT_HAVE_CAS
#define PT_HAVE_CAS2

#include <portablethreads/arch/32-gcc.h>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{

			// The SPARC V8+ architecture/abi is a sun-specific thing which has some of the V9's 
			// instructions as well as 64-bit register support.

			inline bool pt_atomic_set_lock(volatile uint8* lock)
			{
				uint32 result;
				__asm__ __volatile__("ldstub [%1], %0 \n\t"
					: "=r" (result)
					: "r" (lock)
					: "memory");
				return result == 0;
			}

			inline void pt_atomic_clear_lock(volatile uint8* lock)
			{
				__asm__ __volatile__("stb %%g0, [%0] \n\t" : : "r"(lock) : "memory");
			}

			/*
			* Atomic compare and exchange.  Compare OLD with MEM, if identical,
			* store NEW in MEM.  Return the initial value in MEM.  Success is
			* indicated by comparing RETURN with OLD.
			*/
			inline int32 pt_atomic_cas_return_memory(volatile int32* mem, int32 nv, int32 ov)
			{
				__asm__ __volatile__("cas [%2], %3, %0 \n\t"
					"membar #StoreLoad | #StoreStore \n\t"
					: "=&r" (nv)
					: "0" (nv), "r" (mem), "r" (ov)
					: "memory");

				return nv;
			}

			inline int64 pt_atomic_cas_return_memory(volatile int64* mem, int64 nv, int64 ov)
			{
				__asm__ __volatile__
				(
					"mov %1, %%g1 \n\t"
					"ldx [%2], %%g2 \n\t"
					"ldx [%0], %%g3 \n\t"
					"casx [%%g1], %%g2, %%g3\n\t"
					"membar #StoreLoad | #StoreStore \n\t"
					"stx %%g3, [%0] \n\t"
					:
					: "r"(&nv), "r"(mem), "r"(&ov)
					: "g1", "g2", "g3", "memory"
				);
				
				return nv;
			}


			inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}

			inline bool pt_atomic_cas(volatile int64* mem, int64 nv, int64 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}



			inline void pt_barrier()
			{
				__asm__ __volatile__("" : : : "memory");
			}

			inline int32 pt_atomic_set(volatile int32* mem, int32 val)
			{
				__asm__ __volatile__("swap [%2], %0 \n\t"
					: "=&r" (val)
					: "0" (val), "r" (mem)
					: "memory");
				return val;
			}


			// atomically adds value to counter and returns the result
			inline int32 pt_atomic_add(volatile int32* counter, int32 value)
			{
				int32 current;
				do { current = *counter; } while(!pt_atomic_cas(counter, current+value, current));
				return current + value;
			}

			inline int32 pt_atomic_sub(volatile int32* counter, int32 value)
			{
				return pt_atomic_add(counter, -value);
			}

			inline int32 pt_atomic_inc(volatile int32* counter)
			{
				return pt_atomic_add(counter, 1);
			}

			inline int32 pt_atomic_dec(volatile int32* counter)
			{
				return pt_atomic_add(counter, -1);
			}

			inline uint64 pt_ticks()
			{
				uint64 ticks;
				__asm__ 
				(
 					"rd %%tick, %%g1 \n\t"
 					"stx %%g1, [%0] \n\t"
 					:
 					: "r"(&ticks)
 					: "g1"
 				);
			 	
				return ticks;
			}

			const unsigned HARDWARE_POINTER_BITS = 32;
			const unsigned ALIGNMENT_BITS = 2;
		}
	}

	inline uint64 pt_seed()
	{
		return LockFree::Private::pt_ticks();
	}
}

#include <portablethreads/arch/arch-common.h>
#include <portablethreads/arch/native-pointer-cas.h>
#include <portablethreads/arch/native-atomic-number.h>
#include <portablethreads/arch/4-byte-pointer-to-int-compression.h>
#include <portablethreads/arch/free-high-bits-muxer.h>

namespace PortableThreads
{
	namespace LockFree
	{
		namespace Private
		{
			// assume 4 byte alignment
			template<typename T>
			inline int64 pt_inflate_pointer(T* p)
			{
				return pt_inflate_pointer<T, int64, 2>(p);
			}

			template<typename T>
			inline T* pt_deflate_pointer(int64 p)
			{
				return pt_deflate_pointer<T, int64, 2>(p);
			}

			template<unsigned USER_BITS>
			struct PTPointerCASType
			{
				typedef PointerCAS< FreeHighBitsMuxer<int64, 64, 32, 2, USER_BITS> > PTPointerCAS;
			};

			typedef PTPointerCASType<0>::PTPointerCAS PTPointerCAS;
		}
	}
}

#endif
