/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_GRAPH_H
#define LIBNOMORE_GRAPH_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif


#include <color.h>
#include <body_node.h>
#include <head_node.h>
#include <stack.h>
#include <event/channel_manager.h>

#include <iterator>
#include <cassert>


namespace NS_NOMORE {

template <class SkipT>
class SkipIterator 
  : public std::iterator<
      std::forward_iterator_tag, 
      typename SkipT::ResultType
    > {
public:
  typedef std::vector<Node*>::const_iterator base_iterator;
  typedef typename SkipT::ResultType ResultType;
  SkipIterator(base_iterator begin, base_iterator end)
    : begin_(begin)
    , end_(end) {
    skip();
  }
  SkipIterator& operator++() {
    ++begin_;
    skip();
    return *this;
  }
  SkipIterator operator++(int) {
    SkipIterator temp(*this);
    ++*this;
    return temp;
  }
  ResultType operator*() const {
    return SkipT::get(*begin_);
  }

  bool operator==(const SkipIterator& other) const {
    return begin_ == other.begin_;
  }
  bool operator!=(const SkipIterator& other) const {
    return !(*this == other);
  }
  bool operator==(const base_iterator& other) const {
    return begin_ == other;
  }
  bool operator!=(const base_iterator& other) const {
    return !(*this == other);
  }

  base_iterator end() const { return end_; }
private:
  void skip() {
    while (begin_ != end_ && SkipT::skip(*begin_))
      ++begin_;
  }
  base_iterator begin_;
  base_iterator end_;
};

template <class RT, NodeType::Value type>
struct OnlyNodesOfType {
  typedef RT ResultType;
  static bool skip(Node* n) { return n->getType() != type; }
  static ResultType get(Node* n) {
    assert(n->getType() == type);
    return static_cast<ResultType>(n);
  }
};

struct OnlyUncloredNodes {
  typedef Node* ResultType;
  static bool skip(Node* n) { return n->getColor() != Color::none; }
  static ResultType get(Node* n) { return n; }
};

class ChoiceOperator;

//! This class implements nomore's body-head graph. 
/*! The main responsibility of a graph is to store the set of head- and body nodes
 *	of a logic program (edges between the nodes are stored in the nodes not in 
 *  the graph). In addition a graph manages a 
 *  \ref BacktrackStack "backtracking stack" to be used to restore and recolor 
 *  choice points.  
 *	Furthermore a graph is the central place where event handlers register for
 *	specific events.
 *	\see _event
 *
 *	\note The Graph should be used as a static graph, i.e. you construct the graph
 *	in a seperate construction step and once this step is over
 *	you call the Graph's \ref Graph::finishConstruction "finishConstruction" 
 *	method to signal that the graph's structure is now immutable. */
class Graph {
public:
  //! Set of nodes.
  typedef std::vector<Node*> NodeSet;
  typedef NodeSet::size_type size_type;
  
  //! forward iterators for iterating the node set.
  typedef SkipIterator<OnlyNodesOfType<HeadNode*, NodeType::head_node> > HeadNodeIterator;
  typedef SkipIterator<OnlyNodesOfType<BodyNode*, NodeType::body_node> > BodyNodeIterator;
  typedef SkipIterator<OnlyUncloredNodes> UncoloredNodeIterator;

  //! Creates an empty graph and starts its construction phase.
  explicit Graph(BacktrackingMode::Mode mode = BacktrackingMode::symmetric);

  //! Destroys the graph and all contained nodes.
  ~Graph();

  //! Creates and inserts a new body node.
  /*! \pre isConstructionFinished() returns false.
	 *  \return Returns a pointer to the newly created body node.
	 *	\attention The lifetime of the created body node is managed by the graph. */
  BodyNode* insertBodyNode();

  //! Creates and inserts a head node for the atom with the given atomId.
  /*! \pre isConstructionFinished() returns false.
	 *  \return Returns a pointer to a head node with the given id.
	 *	\attention the graph does not check whether a head node for the given atomId
   *  already exists.
   *
   *  \attention The lifetime of the created head node is managed by the graph. */
  HeadNode* insertHeadNode(long atomId);

  
  //! Returns the number of head nodes this graph contains.
	size_type countHeadNodes() const;
	
	//! Returns the number of body nodes this graph contains.
	size_type countBodyNodes() const;

  //! Returns the number of nodes this graph contains.
	size_type countNodes() const {
	  return nodes_.size();
  }

  //! Returns the number of rules this graph contains.
  /*!
   * \note the number of rules is the sum of all body node successors. */
  size_type countRules() const;

  //! Returns an iterator that points to the first head node of this graph.
	/*! If this graph does not have any head nodes the end-iterator is returned.
	 *  \note Traversing the sequence of head nodes yields the nodes in ascending
	 *	      id-order. */
	HeadNodeIterator headNodesBegin() const {
    return HeadNodeIterator(nodes_.begin(), nodes_.end());
  }

  //! Returns an iterator that points just beyond the end of head nodes of this graph.
	HeadNodeIterator headNodesEnd() const {
    return HeadNodeIterator(nodes_.end(), nodes_.end());
  }

  //! Returns an iterator that points to the first head node of this graph.
	/*! If this graph does not have any head nodes the end-iterator is returned. */
	BodyNodeIterator bodyNodesBegin() const {
    return BodyNodeIterator(nodes_.begin(), nodes_.end());
  }

  //! Returns an iterator that points just beyond the end of head nodes of this graph.
	BodyNodeIterator bodyNodesEnd() const {
    return BodyNodeIterator(nodes_.end(), nodes_.end());
  }

  //! Colors a node.
  /*! Calls the method Node::setColor() of the node and fires a color event if 
   *  the node's color was changed.
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \return Returns true if the color has been changed and false otherwise.
	 *	\see Node::setColor
   *  \exception ColorError Throws this exception if a color error occured because 
   *                        of an invalid coloring.
	 *  \post A corresponding stack item was pushed on the backtrack-stack. */
  bool color(Node& n, Color::ColorValue c);

  //! Colors a node as choice point.
  /*! Calls the method Node::setColor() of the node and fires a color choice point 
   *  event if the node's color was changed.
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \param btc The choice-point's alternative color that is used when the choice-point
   *  is recolored 
   *  \return returns true if the color has been changed and false otherwise.
	 *	\see Node::setColor
   *  \see recolorChoicePoint
   *  \exception ColorError Throws this exception if a color error occured because 
   *                        of an invalid coloring.
	 *  \post A new branch was pushed on the backtrack-stack.
   */
  bool colorChoicePoint(Node& n, Color::ColorValue c, Color::ColorValue btc);

  //! Colors a node as choice point that can not be recolored.
  /*! Calls the method Node::setColor() of the node and fires a color choice point 
   *  event if the node's color was changed.
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \return returns true if the color has been changed and false otherwise.
	 *	\see Node::setColor
   *  \exception ColorError Throws this exception if a color error occured because 
   *                        of an invalid coloring.
	 *  \post A corresponding stack item was pushed on the backtrack-stack.
   */
  bool colorChoicePointStable(Node& n, Color::ColorValue c);

  //! Restores a node.
  /*! Calls the method Node::restoreColor() of the node.
   *  \param n The node to be restored.
   *  \param c The new color of the node. Valid parameters are Color::ignore, 
   *           Color::none and Color::weak_plus.
	 *	\see Node::restoreColor 
   */
  void restore(Node& n, Color::ColorValue c);

  
	//! Checks if the graph is total colored.
  /*! That means all nodes (head and body nodes) are colored \ref Color::plus "plus",
   *  \ref Color::minus "minus" or \ref Color::ignore "ignore".
   *  \return True if the graph is total colored else false. */
  bool totalColoring() const {
    // is greater than zero by data type size_t
    // assert(countUncoloredNodes() >= 0);
    return countUncoloredNodes() == 0;
  }

  //! Deletes all nodes of this graph, clears the backtracking stack and starts a new construction phase.
  /*! \note Clearing a graph does not change the graph's
	 *  \ref event::ChannelManager "ChannelManager", i.e. connected handlers stay
	 *	connected. */
  void clear();

  
	//! Returns the number of uncolored nodes this graph contains.
  /*! \note A node counts as uncolored if its color is either Color::none or 
   *        Color::weak_plus. */
	size_type countUncoloredNodes() const {
	  return uncoloredNodes_ + weakColoredNodes_;
  }

  //! Returns the number of weak_plus colored nodes of the graph.
  size_type countWeakColoredNodes() const {
    return weakColoredNodes_;
  }

  //! restores the graph to the state it had just \b after the last choice point was colored.
	/*!
   *  I.e. Restores all nodes that were colored after the last choice point was colored but not the choice point itself.
   * \pre At least one choice point was previously colored.
	 * \see BacktrackStack::restore() 
   */
  void restoreToChoicePoint();

  //!	restores the graph to the state it had just \b before the last choice point was colored.
	/*! 
   *  I.e. restores all nodes that were colored after the last choice point was colored
   *  as well as the last choice point itself.
   * \note The branch that stored the consequences of the last choice point is discarded.
   * \return True if the stack is restored successfully, false if the stack is empty. 
   */
  bool restoreChoicePoint();

  
  //!	first calls restoreLastChoicePoint then recolors the last choice point to its alternative color.	
	/*! Fires a color choice point event. 
   *  If recoloring the last choice points leads to an invalid coloring 
   *  the second last choice point is restored and recolored. This process is repeated
   *  until either the recoloring of a choice point is sucessful or there are
   *  no more choice points left. 
   *  
   *  \return True if the choice point is successfully recolored, false no choice point
   *  could be restored.
   */
  bool recolorChoicePoint(ChoiceOperator& choice);

  
  //! Returns the begin iterator of the uncolored node set.
  UncoloredNodeIterator uncoloredNodesBegin() const {
    return UncoloredNodeIterator(nodes_.begin(), nodes_.end());
  }
  
  //! Returns the end iterator of the uncolored node set.
  UncoloredNodeIterator uncoloredNodesEnd() const {
    return UncoloredNodeIterator(nodes_.end(), nodes_.end());
  }

  //! returns the node with the given id
  /*!
   * \pre 0 <= id < countNodes()
   */
  Node* getNode(long id) const {
    assert(0 <= id && size_type(id) < countNodes());
    return nodes_[id];
  }
  
  //! Returns the \ref event::ChannelManager "ChannelManager" of this graph.
	/*! Interested Event handlers need to register with this 
	 *  \ref event::ChannelManager "ChannelManager" in order to receive events
	 *  concerning this graph. */
	event::ChannelManager& getEventManager() {
    return eventManager_;
  }

	//! Ends the construction phase and fires a GraphConstructionFinished event.
	/*! After calling this method the graph's structure is immutable and
	 *	calls to the insert-methods are prohibited. */
	void finishConstruction();

	//! Returns true, if the graph's construction is completed.
	/*! If this method returns true, the graph's structure can't change anymore.
	 *	If this method returns false, the graph's structure is still volatile. */
	bool isConstructionFinished() const;

  //! totalizes the \ref NS_NOMORE::Graph "graph".
  /*! 
   *  totalize colors all uncolored nodes to Color::minus. 
   *
   *  \note 
   *  This operation is reasonable only if no operator is able to color the 
   *  uncolored nodes.
   *  For example the \ref ForwardPropagator "forward propagator" and
   *  \ref BackwardPropagator "backward propagator" can not always decide whether a 
   *  positve (zero) loop must be colored \ref Color::plus "plus" or 
   *  \ref Color::minus "minus".
   *
   *  \note totalize was formerly implemented as the Post-Operator N.
   *
   *	\note This operator does not verify if the resulting coloring is valid.
   *  This needs to be done in a seperate propagation step (for example using
   *	the \ref ForwardPropagator "forward propagator")
   *
   *  \throw ColorError A ColorError-exception is thrown if the graph contains
   *  Color::weak_plus colored nodes.
   */
  bool totalize();

  BacktrackingMode::Mode getBacktrackingMode() const {
    return backtrackingMode_;
  }

private:
  // graphs currently are not copyable.
  Graph(const Graph& g);
  Graph& operator=(const Graph& g);
  
  inline void colorChanged(Node& n, Color::ColorValue old);
  
  NodeSet nodes_;

  //! Backtracks stack used to restore nodes of this graph.
  BacktrackStack backtrackingStack_;

  BacktrackingMode::Mode backtrackingMode_;

	//! ChannelManager that manages all event channels needed by this graph
	//! and its nodes.
	event::ChannelManager eventManager_;
  
  size_type uncoloredNodes_;
  size_type weakColoredNodes_;

  //! state of the construction phase.
	bool constructionFinished_;
};

//! Output operator for printing the graph to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const Graph& grp);

//! Output operator for printing simple head node to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const HeadNode& head);

//! Output operator for printing simple body node to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const BodyNode& body);

//! Event type fired to signal that the construction of a graph is completed.
/*!  A graph fires this event when it receives the \ref Graph::finishConstruction 
 *   "finishConstruction" message to signal that the graph's structure is
 *   now stable and won't change anymore. */
struct GraphConstructionFinished {
	explicit GraphConstructionFinished(Graph& theGraph)
		: theGraph_(&theGraph) {
	}
	Graph* theGraph_;
};

//! Values for the heuristics stack.
struct HeuristicsStackOperation {
  //! Operation values.
  enum Operation {
    //! push the heuristics stack
    push,
    //! pop the heuristics stack
    pop
  };
};

//! Event type fired to signal that the heuristic stack must be pushed or poped.
/*!  The graph fires this event if the dlv strategie is used and the heuristics
 *   stack must be modified. */
struct HeuristicsStackEvent {
  explicit HeuristicsStackEvent(HeuristicsStackOperation::Operation op)
		: option_(op) {
	}
  HeuristicsStackOperation::Operation option_;
};

} // NS_NOMORE

#endif
