/*	$Id: types.h 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <iostream>
#include <sstream>
#include <algorithm>
#include <platypus/library.h>
#include <platypus/exceptions.h>
#include <platypus/factories/dll_factory.h>
#include <platypus/factories/dirstream/dirstream.h>

using namespace std;

namespace Platypus
{
	/***********************************************/
	/* DLLLoader                                   */
	/***********************************************/
	DLLLoader& DLLLoader::instance()
	{
		static DLLLoader instance_;
		return instance_;
	}
	void DLLLoader::loadAllInDirectory(const std::string& dir, bool report_failure)
	{
		Linkers linkers;

		try
		{
			string filter("*");
			filter += XPLIB_DLL_EXTENSION;
			dirstr::dirstream str(dir.c_str(), dirstr::pattern_f(filter.c_str()));
			if(str.is_open())
			{
				string entry;
				while(str >> entry)
				{
					try
					{
						toolbox::StackPtr<XPDLLLinker> linker(load(entry, true));
						if(linker.get())
						{
							linkers.push_back(linker);
						}
					}
					catch(const XPDLLException& e)
					{
						if(report_failure)
						{
							ostringstream s;
							s	<< "Failed to open shared library '"
								<< entry 
								<< "'. Reason: " 
								<< e.what();
							throw PlatypusException(s.str());
						}
					}
				}
			}
		}
		catch(const dirstr::open_error&)
		{}
		
		for(Linkers::iterator newit = linkers.begin(); newit != linkers.end(); ++newit)
		{
			bool foundit = false;
			for(Linkers::iterator oldit = linkers_.begin(); oldit != linkers_.end(); ++oldit)
			{
				if(**oldit == **newit)
				{
					foundit = true;
					break;
				}
			}
			if(!foundit)
			{
				initializeLibrary(**newit);
				linkers_.push_back(*newit);
			}
		}
	}
	void DLLLoader::initializeLibrary(XPDLLLinker& linker)
	{
		XPFunctionProxy<platypus_dll_initialize_function> ctor(
			linker.reference(PLATYPUS_DLL_INITIALIZE));
		assert(ctor.good());
		ctor();
	}
	
	DLLLoader::const_iterator DLLLoader::begin()
	{
		return linkers_.begin();
	}
	DLLLoader::const_iterator DLLLoader::end()
	{
		return linkers_.end();
	}
	
	bool DLLLoader::hasProperFunctions(XPDLLLinker& linker)
	{
		return	linker.reference(PLATYPUS_DLL_INITIALIZE) &&
				linker.reference(PLATYPUS_DLL_FINALIZE);
	}
	XPDLLLinker* DLLLoader::load(const std::string& path, bool complete)
	{
		toolbox::SmartPtr<XPDLLLinker> linker(new XPDLLLinker(path, complete));
		if(!hasProperFunctions(*linker))
		{
			linker.reset();
		}
		
		return linker.release();
	}

	DLLLoader::DLLLoader()
	{
		
	}
	DLLLoader::~DLLLoader()
	{
		for(Linkers::const_iterator it = linkers_.begin(); it != linkers_.end(); ++it)
		{
			XPFunctionProxy<platypus_dll_finalize_function> dtor(
				(*it)->reference(PLATYPUS_DLL_FINALIZE));
			assert(dtor.good());
			dtor();
		}		
	}
}


