/*  Copyright (c) March 2006 Jean Gressmann (jean@0x42.de)
*
*  This is free software; you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation; either version 2 of the License, or
*	(at your option) any later version. 
* 
*	This file is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this file; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef X86_32_GCC_COMMON_H
#define X86_32_GCC_COMMON_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			inline void pt_atomic_clear_lock(volatile uint8* lock)
			{
				__asm__ __volatile__(
					"lock; cmpxchgb %b0, %1 \n\t"
					:
					: "q"(0), "m"(*lock), "a"(0xff)
					: "memory"
					);
			}

			inline bool pt_atomic_set_lock(volatile uint8* lock)
			{
				uint8 prev;
				__asm__ __volatile__(
					"lock; cmpxchgb %b1, %2 \n\t"
					: "=a"(prev)
					: "q"(0xff), "m"(*lock), "0"(0)
					: "memory"
					);

				return prev == static_cast<uint8>(0);
			}

			inline void pt_mfence()
			{
				// save ebx in esi as the register will be needed for PIC code
				__asm__ __volatile__(
					"movl %%ebx, %%esi\n\t"
					"cpuid\n\t"
					"movl %%esi, %%ebx\n\t"
					:
					: "a"(0)
					: "memory", "edx", "esi", "ecx"
				);
			}

			inline int32 pt_atomic_add(volatile int32* counter, int32 value)
			{
				const int32 res = value;
				__asm__ __volatile__(
					"lock; xaddl %0, %1	\n\t"
					:"=r"(value)
					:"m"(*counter), "0"(value)
					);
				return res + value;
			}

			

			/*
			From Intel Pentium Manual: cmpxchg
			Operation
			(* accumulator = AL, AX, or EAX, depending on whether *)
			(* a byte, word, or doubleword comparison is being performed*)
			IF accumulator = DEST
			THEN
				ZF = 1
				DEST = SRC
			ELSE
				ZF = 0
				accumulator = DEST
			FI;
			*/
			inline int32 pt_atomic_cas_return_memory(volatile int32* inMemory, int32 newValue, int32 oldValue)
			{
				int32 prev;	
				__asm__ __volatile__(
					"lock; cmpxchgl %1, %2	\n\t"
					:	"=a"(prev)
					:	"q"(newValue), "m"(*inMemory), "0"(oldValue)
					:	"memory"
					);
				return prev;
			}

			inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}
			
			inline int32 pt_atomic_set(volatile int32* inMemory, int32 newValue)
			{
				// This always asserts a lock (Pentium Manual)
				__asm__ __volatile__(
					"xchgl %0, %1	\n\t"
					:	"=r"(newValue)
					:	"m"(*inMemory), "0"(newValue)
					:	"memory"
					);
				return newValue;
			}

			// prototype so that later functions compile
			inline uint64 pt_ticks();
		}
		
	}

	inline uint64 pt_seed()
	{
		return LockFree::Private::pt_ticks();
	}
}


#endif
