/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef X86_32_GCC_H
#define X86_32_GCC_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#define PT_HAVE_CAS
#define PT_HAVE_CAS2

#include <portablethreads/arch/32-gcc.h>
#include <portablethreads/arch/x86-32-gcc-common.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-add.h>
#include <portablethreads/arch/cas-based-64-bit-atomic-set.h>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			// 64 bit
			inline bool pt_atomic_cas(volatile int64* mem, int64 newValue, int64 oldValue)
			{
				/* (from Pentium Manual)
				Compares the 64-bit value in EDX:EAX with the operand (destination operand). If the values
				are equal, the 64-bit value in ECX:EBX is stored in the destination operand. Otherwise, the
				value in the destination operand is loaded into EDX:EAX. The destination operand is an 8-byte
				memory location. For the EDX:EAX and ECX:EBX register pairs, EDX and ECX contain the
				high-order 32 bits and EAX and EBX contain the low-order 32 bits of a 64-bit value.
				*/

				// NOTE: This code is only so complicated b/c in PIC mode we cannot
				// use the ebx register, not even in the clobber list. However, b/c
				// cmpxchg8b REQUIRES the ebx register we need to store its content
				// first, do the op and then restore ebx to its previous content.
				int32 buf[] = {	static_cast<int32>(oldValue),
								static_cast<int32>(oldValue >> 32),
								static_cast<int32>(newValue),
								static_cast<int32>(newValue >> 32),
								0,
								0};
				__asm__ __volatile__(
					"movl %%ebx, 16(%0)		\n\t"
					"movl 0(%0), %%eax		\n\t"
					"movl 4(%0), %%edx		\n\t"
					"movl 8(%0), %%ebx		\n\t"
					"movl 12(%0), %%ecx		\n\t"
					"lock; cmpxchg8b (%1)   \n\t"
					"sete 20(%0)			\n\t"
					"movl 16(%0), %%ebx		\n\t"
					:
					: "S"(buf), "D"(mem)
					: "eax", "edx", "ecx", "cc", "memory"
				);

				return buf[5] != 0; 
			}

			inline uint64 pt_ticks()
			{
				/*
				Loads the current value of the processors time-stamp counter into the EDX:EAX registers. The
				time-stamp counter is contained in a 64-bit MSR. The high-order 32 bits of the MSR are loaded
				into the EDX register, and the low-order 32 bits are loaded into the EAX register.
				*/
				uint32 low, high;
				__asm__ (
					"rdtsc			\n\t"
					:"=a"(low), "=d"(high)
					:
					/* empty clobber list */
				);
				
				uint64 tsc = high;
				tsc <<= 32;
				tsc |= low;
				return tsc; 
			}

			const unsigned HARDWARE_POINTER_BITS = 32;
			const unsigned ALIGNMENT_BITS = 2;
		}
	}
}

#include <portablethreads/arch/arch-common.h>
#include <portablethreads/arch/native-atomic-arithmetic-common.h>
#include <portablethreads/arch/native-pointer-cas.h>
#include <portablethreads/arch/native-atomic-number.h>
#include <portablethreads/arch/free-high-bits-muxer.h>

namespace PortableThreads
{
	namespace LockFree
	{
		namespace Private
		{
			template<unsigned USER_BITS, unsigned ALIGNMENT = ALIGNMENT_BITS>
			struct PTPointerCASType
			{
				typedef PointerCAS< FreeHighBitsMuxer<int64, 64, HARDWARE_POINTER_BITS, ALIGNMENT, USER_BITS> > PTPointerCAS;
			};

			typedef PTPointerCASType<0>::PTPointerCAS PTPointerCAS;
		}
	}
}

#endif
