/*  Copyright (c) Febuary 2006 Jean Gressmann (jean@0x42.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_LOCK_FREE_HEAP_H
#define PT_LOCK_FREE_HEAP_H
#include <portablethreads/config.h>
#include <portablethreads/tls.h>
#include <portablethreads/lockfree/stack.h>
#include <cstddef>

namespace PortableThreads 
{
	namespace LockFree
	{
		/*! \class PTHeap heap.h portablethreads/lock_free.h
		\brief Lock-free heap implementation based on Doug Lea's malloc
			which is available from ftp://gee.cs.oswego.edu/pub/misc/malloc.c


		Each PTHeap object manages memory individually for each thread which 
		accesses the heap. Only those threads may use a given PTHeap object
		that where started (e.g. via PThread::run()) AFTER the heap object was 
		created. Upon destruction the PTHeap object frees all memory 
		allocated. Access to memory optained via a given PTHeap object and 
		referenced after the object's destruction results in undefined behavior.

		There is a default heap (PTHeap::defaultHeap()) which can be used with any 
		thread in the system as it is created before the main()-function is executed.

		*/
		class PTHeap
		{
		public:
			~PTHeap();
			/*!	\brief Creates a heap object with its private memory pool

			\exception Raises PTResourceError if no thread local storage slot can be obtained.
			*/
			explicit
			PTHeap();
			/*!	\brief Allocate memory

			\param size Number of bytes to allocate.
			\exception Throws std::bad_alloc if no memory can be obtained from the system.
			*/
			void* allocate(std::size_t size);
			/*! \brief Free previously allocated memory

			Note that the thread doing the deallocated isn't required to be the same thread
			that allocated the memory.
			*/
			void deallocate(void* p);
			//! Default heap which is used with PTAllocator and PTUseDefaultHeapMixin
			static PTHeap& defaultHeap();
		private:
			PTHeap(const PTHeap&);
			PTHeap& operator=(const PTHeap&);			
		private:
			typedef PTStack<void*> Heaps;
			Heaps heaps_;
			const PThreadLocalStorage::key_type key_;
		};
	}
}


#endif

