/*  Copyright (c) October 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_LOCK_FREE_UTILITY_H
#define PT_LOCK_FREE_UTILITY_H
#include <portablethreads/config.h>
#include <cassert>

#ifdef _MSC_VER
// This warning also appears when a 32-bit pointer gets cast into a 4 byte int
#	pragma warning(disable:4311) // A 64-bit pointer was truncated to a 32-bit int or 32-bit long.
// This warning also appears when a 32-bit int gets cast into a 32-bit pointer
#	pragma warning(disable:4312) // You attempted to assign a 32-bit value to a 64-bit integer. For example, casting a 32-bit int or 32-bit long to a 64-bit pointer
#endif

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			template<typename CASType, typename T>
			inline void stackPush(CASType& head, T* node)
			{
				assert(sizeof(T*) <= sizeof(typename CASType::int_t));

				typename CASType::token_t ov, nv;
				do
				{
					ov = head.get();
					node->next(ov);
					nv.pointer(reinterpret_cast<typename CASType::int_t>(node));
					nv.userbits(ov.userbits());
				}
				while(!head.cas(nv, ov));
			}

			// NOTE: this function does NOT guarantee that node has
			// the same value as before when returning unsuccessfully!
			template<typename CASType, typename T>
			inline bool stackPop(CASType& head, T*& node)
			{
				assert(sizeof(T*) <= sizeof(typename CASType::int_t));

				for(typename CASType::token_t h = head.get(), nh; h.pointer(); h = head.get())
				{
					nh = reinterpret_cast<T*>(h.pointer())->next();
					if(head.cas(nh, h))
					{
						node = reinterpret_cast<T*>(h.pointer());
						return true;
					}
				}
				return false;
			}

			class PTStackable
			{
			public:
				typedef PTPointerCAS::token_t token_t;
				typedef PTPointerCAS::token_t::int_t int_t;
				template<typename T>
				inline void push(T* p)
				{
					stackPush<PTPointerCAS, T>(next_, p);
				}
				template<typename T>
				inline T* pop()
				{
					T* ret = 0;
					if(stackPop<PTPointerCAS, T>(next_, ret))
						return ret;
					return 0;
				}
				template<typename T>
				inline T* top() 
				{
					reinterpret_cast<T*>(next_.get().pointer());
				}
				inline token_t next() const { return next_.get(); }
				inline void next(int_t p, int_t u = 0) { next_.assign(p, u); }
				inline void next(const token_t& t) { next_.assign(t); }
				inline bool cas(token_t& nv, const token_t& ov)
				{
					return next_.cas(nv, ov);
				}
			protected:
				~PTStackable() {}
				PTStackable() {}				
			private:
				PTPointerCAS next_;
			};

		}
	}

}

#ifdef _MSC_VER
#	pragma warning(default:4311)
#	pragma warning(default:4312)
#endif

#endif

