/*  Copyright (c) Febuary 2006 Jean Gressmann (jean@0x42.de)
*
*  This is free software; you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation; either version 2 of the License, or
*	(at your option) any later version. 
* 
*	This file is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this file; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef PT_TLS_H
#define PT_TLS_H
#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/tls.h>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/tls.h>
#endif

namespace PortableThreads
{
	/*! \struct PThreadLocalStorage tls.h portablethreads/tls.h
	\brief TLS

	PThreadLocalStorage allows to allocate memory slots which are unique for each thread
	in the system. Each slot is references using a key. The value stored for each slot
	is local to the thread (thread local storage, TLS).
	*/
	struct PThreadLocalStorage : public OSSpecific::PThreadLocalStorage
	{
		typedef OSSpecific::PThreadLocalStorage::key_type key_type; //!< key type used to reference a slot (pod)

		/*! \brief Creates a slot

		The pair contains the slot key as first member and
		the signals success of the operation in the second.

		After a successful call each thread in the process can subsequently
		use the slot key to access the its private slot.

		\exception Raises PTResourceError if no TLS key can be obtained.
		*/
		static inline key_type create()
		{
			key_type key;
			if(!OSSpecific::PThreadLocalStorage::create(key))
				throw PTResourceError("[PThreadLocalStorage] Could not allocate thread local storage slot");
			return key;
		}
		/*! \brief Destroys a slot

		Note that if the slot holds a pointer to a heap memory this 
		memory will not be freed.
		*/
		static inline void destroy(key_type id)
		{
			return OSSpecific::PThreadLocalStorage::destroy(id);
		}
		/*! \brief Sets the the slot to the specified value

		Returns true on success.
		*/
		static inline bool set(key_type id, void* p)
		{
			return OSSpecific::PThreadLocalStorage::set(id, p);
		}
		/*! \brief Gets the current slot value

		Returns the current value if the slot identifier is valied.
		Otherwise zero is returned.
		*/
		static inline void* get(key_type id)
		{
			return OSSpecific::PThreadLocalStorage::get(id);
		}
	};
}



#endif
