/*  Copyright (c) Febuary 2006 Jean Gressmann (jean@0x42.de)
*
*  This is free software; you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation; either version 2 of the License, or
*	(at your option) any later version. 
* 
*	This file is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this file; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <portablethreads/exception.h>
#include <portablethreads/lockfree/heap.h>
#include <portablethreads/mutex.h>
#include <portablethreads/utility.h>
#include <cassert>
#include <algorithm>
#include <utility>
#include "malloc.h"

using namespace std;

namespace PortableThreads
{
	namespace LockFree
	{
		namespace
		{
			struct OutOfMemory : public std::bad_alloc
			{
				const char *what() const throw()
				{
					return "[PTHeap] Memory allocation failed";
				}
			};

			// align to double
			const unsigned MMGT_SIZE = 8;
			typedef std::pair<void*, PTMutex> MallocState;
		}
		PTHeap::~PTHeap()
		{
			for(void* p = 0; heaps_.pop(p); )
			{
				MallocState* state = reinterpret_cast<MallocState*>(p);
				destroy_mspace(state->first);
				delete state;
			}
			PThreadLocalStorage::destroy(key_);
		}
		
		PTHeap::PTHeap()
			:	key_(PThreadLocalStorage::create())
		{}
		void* PTHeap::allocate(size_t size)
		{
			MallocState* state = static_cast<MallocState*>(PThreadLocalStorage::get(key_));
			if(!state)
			{
				void* space = create_mspace(0, 0);
				if(!space)
					throw OutOfMemory();
				
				bool gotState = false;
				try
				{
					state = new MallocState;
					gotState = true;
					state->first = space;
					heaps_.push(state);
				}
				catch(...)
				{
					if(gotState)
						delete state;
					destroy_mspace(space);
					throw;
				}
				
				PThreadLocalStorage::set(key_, state);
			}
			
			void* allocated = 0;
			{
				PTGuard<PTMutex> guard(state->second);
				allocated = mspace_malloc(state->first, size + MMGT_SIZE);
			}
			
			if(!allocated)
				throw OutOfMemory();
			*static_cast<void**>(allocated) = state;
			return static_cast<char*>(allocated) + MMGT_SIZE;
		}
		void PTHeap::deallocate(void* p)
		{
			if(p)
			{
				void* allocated = static_cast<char*>(p) - MMGT_SIZE;
				assert(*static_cast<void**>(allocated));
				MallocState* state = static_cast<MallocState*>(*static_cast<void**>(allocated));
#ifndef NDEBUG
				const bool found = find(heaps_.begin(), heaps_.end(), state) != heaps_.end();
				assert(found && "[PTHeap] You tried to free memory with the wrong heap object");		
#endif

				{
					PTGuard<PTMutex> guard(state->second);
					mspace_free(state->first, allocated);
				}
			}
		}
		PTHeap& PTHeap::defaultHeap()
		{
			static PTHeap default_;
			return default_;
		}

		static const PTHeap* __dummy = &PTHeap::defaultHeap();
	}
}


