#ifndef THREAD_NET_H
#define THREAD_NET_H

#include <memory>
#include <cassert>
#include <cstring>
#include <portablethreads/thread.h>
#include <portablethreads/semaphore.h>
#include <portablethreads/lockfree/allocator.h>
#include <portablethreads/lockfree/queue.h>

namespace PortableThreads
{
	namespace Private
	{
		template<typename MessageType>
		class PThreadGridShared
		{
		public:
			~PThreadGridShared()
			{
				MessageType* m;
				for(unsigned i = 0; i < count_; ++i)
				{
					while(channels_[i].first.popFront(m))
						delete m;
				}
				delete [] channels_;
			}
			PThreadGridShared(unsigned c)
				:	channels_(new MessageChannel[c])
				,	count_(c)
			{}
			inline unsigned count() const
			{
				return count_;
			}
			void send(unsigned to, std::auto_ptr<MessageType> msg)
			{
				assert(to < count_);
				assert(msg.get());
				channels_[to].first.pushBack(msg.release());
				channels_[to].second.up();
			}
			void broadcast(std::auto_ptr<MessageType> msg)
			{
				assert(count_);
				assert(msg.get());
				for(unsigned i = 1; i < count_; ++i)
				{
					MessageType* c = msg->clone();
					assert(c && "[PThreadGrid] PTClonableGridMessage::clone() implementation returned 0");
					channels_[i].first.pushBack(c);
					channels_[i].second.up();
				}
				channels_[0].first.pushBack(msg.release());
				channels_[0].second.up();
			}
			std::auto_ptr<MessageType> wait(unsigned id)
			{
				assert(id < count_);
				channels_[id].second.down();
				MessageType* m = 0;
#ifdef NDEBUG
				channels_[id].first.popFront(m);
#else
				const bool ok = channels_[id].first.popFront(m);
				assert(ok);
#endif
				assert(m);
				return std::auto_ptr<MessageType>(m);
			}
			bool query(unsigned id, std::auto_ptr<MessageType>& msg)
			{
				assert(id < count_);
				const bool ret = channels_[id].second.tryDown();
				if(ret)
				{
					MessageType* m = 0;
					channels_[id].first.popFront(m);
					assert(m);
					msg.reset(m);
				}
				return ret;
			}
		private:
			typedef LockFree::PTQueue< MessageType*, LockFree::PTAllocator<MessageType*> > Queue;
			typedef std::pair<Queue, PTSemaphore> MessageChannel;
		private:
			MessageChannel* channels_;
			unsigned count_;
		};		
	}

	template<typename T> class PThreadGrid;

	template<typename MessageType>
	class PTGridWorker : private PThread
	{
		friend class PThreadGrid<MessageType>;
	protected:
		PTGridWorker()
			:	shared_(0)
		{}
		inline unsigned workerId() const
		{
			assert(shared_);
			return id_;
		}
		inline unsigned workerThreads() const
		{
			assert(shared_);
			return shared_->count();
		}
		inline void send(unsigned to, std::auto_ptr<MessageType> msg)
		{
			assert(msg.get());
			assert(shared_);
			shared_->send(to, msg);
		}
		inline void send(unsigned to, MessageType* msg)
		{
			send(to, std::auto_ptr<MessageType>(msg));
		}
		inline void broadcast(MessageType* msg)
		{
			broadcast(std::auto_ptr<MessageType>(msg));
		}
		inline void broadcast(std::auto_ptr<MessageType> msg)
		{
			assert(msg.get());
			assert(shared_);
			shared_->broadcast(msg);
		}
		inline std::auto_ptr<MessageType> wait()
		{
			assert(shared_);
			return shared_->wait(id_);
		}
		inline bool query(std::auto_ptr<MessageType>& msg)
		{
			assert(shared_);
			return shared_->query(id_, msg);
		}
	private:
		inline void shared(Private::PThreadGridShared<MessageType>* s)
		{
			assert(s);
			shared_ = s;
		}
		inline void id(unsigned i)
		{
			id_ = i;
		}
	private:
		Private::PThreadGridShared<MessageType>* shared_;
		unsigned id_;
	};

	
	

	template<typename MessageType>
	class PThreadGrid
	{
	public:
		typedef PTGridWorker<MessageType> Worker;
		~PThreadGrid()
		{
			delete [] threads_;
		}
		PThreadGrid(unsigned count)
			:	shared_(count)
			,	threads_(new Worker*[count])
		{
			std::memset(threads_, 0, sizeof(Worker*)*count);
		}

		inline unsigned workerThreads() const
		{
			return shared_.count();
		}
		void setWorker(unsigned id, Worker& thread)
		{
			assert(id < shared_.count());
			assert(PThread::poll(thread.threadId()) != PThread::RUNNING);
			threads_[id] = &thread;
			threads_[id]->id(id);
			threads_[id]->shared(&shared_);
		}
		inline void broadcast(MessageType* msg)
		{
			broadcast(std::auto_ptr<MessageType>(msg));
		}
		inline void broadcast(std::auto_ptr<MessageType> msg)
		{
			if(shared_.count())
				shared_.broadcast(msg);
		}
		inline void send(unsigned to, std::auto_ptr<MessageType> msg)
		{
			assert(to < shared_.count());
			shared_.send(to, msg);
		}
		inline void send(unsigned to, MessageType* msg)
		{
			send(to, std::auto_ptr<MessageType>(msg));
		}
		void run()
		{
			for(unsigned i = 0; i < shared_.count(); ++i)
			{
				if(threads_[i])
					threads_[i]->run();
			}
		}
		void join()
		{
			for(unsigned i = 0; i < shared_.count(); ++i)
			{
				if(threads_[i])
					threads_[i]->join();
			}
		}
	private:
		PThreadGrid();
		PThreadGrid(const PThreadGrid&);
		PThreadGrid& operator=(const PThreadGrid&);
	private:
		Private::PThreadGridShared<MessageType> shared_;
		Worker** threads_;
	};	
}

#endif
