/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_SEQUENCE_OPERATOR_H
#define NOMORE_SEQUENCE_OPERATOR_H
#include <operator.h>
#include <vector>
namespace NS_NOMORE {

//! A composite operator that combines several operators into one and executes them in sequence.
/*!
 * If closure is set to true, the sequence is repeatedly executed as long
 * as at least one contained operator returns true.
 */
class Sequence : public Operator {
public:
  //! creates an empty sequence
  /*!
   * \param closure true if the sequence should be executes as long as at least 
   * one contained operator returns true.
   */
  explicit Sequence(bool closure = false);
  
  //! creates a sequence containing the operators o1 and o2
  /*!
   * \param o1 the first operator in this sequence
   * \param o2 the second operator in this seqeunce
   * \param closure true if the sequence should be executes as long as at least 
   * one contained operator returns true.
   *
   * \pre o1 and o2 are both != 0 and both operators were allocated using new.
   * \note The sequence takes over ownership of the operators.
   */
  Sequence(Operator* o1, Operator* o2, bool closure = false);
  
  //! destroys the sequence and all contained operators
  ~Sequence();
	
	//! adds the operator o to this sequence
  /*!
   * \pre o was allocated using new
   * \note the squence takes over ownership of the given operator.
   */
  void add(Operator* o);

  //! adds the operator o to this sequence without transfering ownership to the sequence.
  void add(Operator& o);

  //! adds the operator o to this sequence
  /*!
   * \param o the operator to be added
   * \param takeOverOwnership true if the sequence should take over ownership
   * of the given operator
   * \pre o != 0. If takeOverOwnership is true, o must point to a dynamically
   * allocated object.
   */
  void add(Operator* o, bool takeOverOwnership);
	
	//! sets the closure-property of this sequence.
	void setClosure(bool b);

  //! returns the operator with name name if such an operator is contained in this sequence.
  virtual Operator* extractOperator(const std::string& name);

  //! returns true if ops is valid for all contained operators.
  bool validate(const OpSet& ops) const;
private:
	//!	executes all operators contained in this objec in sequence.
	/*!	If closure is set to true, this method keeps executing the
	 *	operators until all return false.
	 *	\return true if at least one operator of this sequence returned true and
	 *	closure is false. Otherwise false.
	 */
	bool execute();
  typedef std::pair<Operator*, bool> OpInfo;
  typedef std::vector<OpInfo> Operators;
  Operators ops_;
	bool closure_;
};

}
#endif
