/*	$Id: types.h 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <platypus/types/search_space.h>

namespace Platypus
{
	////////////////////////////////////////////////////////////////////
	// detail::Node
	////////////////////////////////////////////////////////////////////
	namespace detail
	{
		Node::Node(ChoiceId id)
		:	id_(id)
		,	active_(false)
		{}
		ChoiceId Node::id() const
		{
			return id_;
		}
		void Node::active(bool b)
		{
			active_ = b;
		}
		bool Node::active() const
		{
			return active_;
		}
	}
	
	////////////////////////////////////////////////////////////////////
	// SearchSpace
	////////////////////////////////////////////////////////////////////
	SearchSpace::SearchSpace(const Branch& branch)
		:	empty_(false)
	{
		space_.root(node_type(static_cast<ChoiceId>(-1)));
		integrate(branch);
	}
	SearchSpace::cursor SearchSpace::close(cursor c)
	{
		const cursor ROOT = space_.root();
		for(cursor parent = c.parent(); 
			c != ROOT && !c.has_left_child() && !c.has_right_child(); 
			parent.ascend())
		{
			space_.erase(c);
			c = parent;
		}
		empty_ = c == ROOT && !c.has_left_child() && !c.has_right_child();
		return c;
	}
	bool SearchSpace::empty() const
	{
		return empty_;
	}
	SearchSpace::cursor SearchSpace::add(cursor c, const Choice& choice)
	{
		space_.insert_left(c, node_type(choice.id()));
		space_.insert_right(c, node_type(choice.id()));
		
		if(choice.isPositive())
			c.descend_left();
		else
			c.descend_right();
		return c;
	}
	SearchSpace::cursor SearchSpace::integrate(const Branch& branch)
	{
		empty_ = false;
		if(branch.empty())
			return space_.root();

		return integrate(space_.root(), branch.begin(), branch.end());
	}
	void SearchSpace::findLastCommonNode(cursor& c, Branch::const_iterator& it, Branch::const_iterator END)
	{
		// move down the tree as far as possible
		assert(c.good());
		for(; it != END; ++it)
		{
			if(it->isPositive())
			{
				if(c.has_left_child())
					c.descend_left();
				else
					break;
			}
			else
			{
				if(c.has_right_child())
					c.descend_right();
				else
					break;
			}
		}
	}
	SearchSpace::cursor SearchSpace::integrate(cursor c, Branch::const_iterator it, Branch::const_iterator END)
	{
		findLastCommonNode(c, it, END);

		// make sure this branch isn't a true subbranch
		assert(it != END);

		
		// do insert, beginning one past the
		// last common element
		for(; it != END; ++it)
		{
			if(it->isPositive())
			{
				c = space_.insert_left(c, node_type(it->id()));
			}
			else
			{
				c = space_.insert_right(c, node_type(it->id()));
			}
		}
		return c;
	}
	SearchSpace::cursor SearchSpace::root() const
	{
		return const_cast<tree_type&>(space_).root();
	}
	Branch SearchSpace::stackToBranch() const
	{
		Branch b;
		b.reserve(speedup_stack_.size());
		while(!speedup_stack_.empty())
		{
			b.push_back(speedup_stack_.back());
			speedup_stack_.pop_back();
		}
		return b;
	}
	void SearchSpace::packStack(cursor from, cursor to) const
	{
		assert(speedup_stack_.empty());
		for(cursor parent = from.parent(); from != to; parent.ascend())
		{
			speedup_stack_.push_back(Choice(from->id(), parent.left() == from));
			from = parent;
		}
	}
	Branch SearchSpace::extract(cursor c)
	{
		if(	!c.good() || 
			c.has_left_child() || c.has_right_child())
			return Branch();
		
		assert(c != space_.root());
        packStack(c, space_.root());
		close(c);
		return stackToBranch();
	}
	Branch SearchSpace::branch(cursor l, cursor h) const
	{
		if(!l.good() || !h.good())	
			return Branch();
		
		packStack(l, h);

		return stackToBranch(); 
	}
	Branch SearchSpace::branch(cursor lower) const
	{
		return branch(lower, const_cast<tree_type&>(space_).root());
	}
	double SearchSpace::size() const
	{
		if(empty_)
			return 0;
		
		return size(space_.root(), 100);
	}

	double SearchSpace::size(const_cursor c, double value)
	{
		const double leftSize = c.has_left_child() ? size(c.left(), value / 2) : 0;
		const double rightSize = c.has_right_child() ? size(c.right(), value / 2) : 0;
		return leftSize + rightSize;
	}

	SearchSpace::leaf_iterator SearchSpace::leafBegin()
	{
		return space_.begin<leafs_only>();
	}
	SearchSpace::leaf_iterator SearchSpace::leafEnd()
	{
		return space_.end<leafs_only>();
	}
}

