#include <iostream>
#include <string>
#include <cstring>
#include <sstream>
#include <stdexcept>
#include "directory.h"
#include "program_options.h"
#include "value.h"
#include "system.h"
#include "path.h"
#include "string_functions.h"
#include "lowercase.h"
#include "config.h"

#ifdef PT_HAVE_LINUX // set in configure.in
	#include <gnu/libc-version.h>
#endif

using namespace std;
using namespace ProgramOptions;

unsigned WIDTH = 32;
const char* const LINUX = "linux";
const char* const SUNOS = "sunos";
const char* const MINGW32 = "mingw32";
string lib(lib_path()), include(include_path());

system_information_t system_description;

bool contains(const std::string& haystack, const char* needle);
bool contains(const char* haystack, const char* needle);
void setPaths(const OptionValues& values);
void setMode(const OptionValues& values);
bool setSystem(const OptionValues& values, system_information_t& info);

void libs(bool own, bool shared, std::ostream& os, const system_information_t& system_description);
void defines(std::ostream& os, const system_information_t& system_description);
void osSpecificDefines(std::ostream& os, const system_information_t& system_description);
void cxxflags(std::ostream& os, const system_information_t& system_description);
void ldflags(std::ostream& os, const system_information_t& system_description);
void completeSystemInformation(system_information_t& info);

int main(int argc, char** argv)
{
	OptionGroup options("This is the PortableThreads configuration utility.\nIt provides command line options to compile and link with PortableThreads.");
	options.addOptions()
		("help,h", bool_switch()->defaultValue(false), "print this")
		("defines", bool_switch()->defaultValue(false), "print defines to compile")
		("cxxflags", bool_switch()->defaultValue(false), "print flags for g++ to compile")
		("ldflags", bool_switch()->defaultValue(false), "print flags for g++ to link")
		("libs", bool_switch()->defaultValue(false), "print libs to link with")
		("lib-dir", value<string>(), "override default path for libraries")
		("include-dir", value<string>(), "override default path for header files")
		("no-own-libs", bool_switch()->defaultValue(false), "just put non-PortableThreads libraries with --libs")
		("shared", bool_switch()->defaultValue(false), "use shared libraries")
		("mode", value<string>()->defaultValue("detect"), "use 32 or 64 bit mode, default: detect")
		("system", value<string>(), "Provide flags for system (GNU build-string), default: current")
	;
	
	try
	{
		OptionValues values(parseCommandLine(argc, argv, options, true));
		if(option_as<bool>(values, "help"))
		{
			cout << options << endl;
			return 0;
		}

		// gather system information
		const bool useSystemString = setSystem(values, system_description);
		if(!useSystemString)
			getSystemInformation(system_description);
		completeSystemInformation(system_description);

		setPaths(values);
		setMode(values);
		

		ostringstream s;
		if(option_as<bool>(values, "defines"))
		{
			defines(s, system_description);
			if(!useSystemString)
				osSpecificDefines(s, system_description);
		}
		if(option_as<bool>(values, "cxxflags"))
			cxxflags(s, system_description);

		if(option_as<bool>(values, "ldflags"))
			ldflags(s, system_description);

		if(option_as<bool>(values, "libs"))
			libs(!option_as<bool>(values, "no-own-libs"), option_as<bool>(values, "shared"),
				s, system_description);

		cout << s.str() << endl;;
	}
	catch(const exception& e)
	{
		cout << e.what() << endl;
		return 1;
	}
	catch(...)
	{
		cout << "unknown error, exiting" << endl;
		return 1;
	}

	return 0;
}


bool setSystem(const OptionValues& values, system_information_t& info)
{
	const bool useSystemString = values.count("system") > 0;
	if(useSystemString)
	{
		/*
			IA64, Linux
				ia64-*-linux-gnu
			IA32, Linux
				i586-pc-linux-gnu
				i686-pc-linux-gnu
			IA32, XP
				i686-pc-mingw32
			SPARC, V9, Solaris
				sparc-sun-solaris2.9
			IA32, Solaris
				i386-pc-solaris2.10
		*/
		string system = option_as<string>(values, "system");
		lowercase(system);
		const vector<string> tokens = StringFunctions::explode("-", system);
		if(tokens.size() == 3 || tokens.size() == 4)
		{
			info.machine_ = tokens[0];
			if(contains(tokens[2], "solaris") || contains(tokens[2], "sunos"))
			{
				info.os_ = SUNOS;
				if(tokens[0] == "i386")
					info.machine_ = "i586";
			}
			else if(contains(tokens[2], "linux"))
			{
				info.os_ = LINUX;
			}
			else if(contains(tokens[2], "mingw32"))
			{
				info.os_ = MINGW32;
			}
		}
	}
	return useSystemString;
}


void completeSystemInformation(system_information_t& info)
{
	// On Linux system, 
	if(contains(info.os_, LINUX))
	{
		if(contains(info.machine_, "sparc")) // if the CPU is a SPARC set machine details
		{
			// V8
			info.machine_details_ += " sparcv8";
			if(contains(info.machine_, " sun4u")) // V9
			{
				info.machine_details_ += "sparcv8plus sparcv9";
			}
		}
		else if(contains(system_description.machine_, "x86_64")) 
		{
			// assume amd64
			info.machine_ += " i86pc";
			info.machine_details_ += " amd64";
		}
		else if(contains(system_description.machine_, "i586") ||
				contains(system_description.machine_, "i686"))
		{
			info.machine_ += " i86pc";
		}
	}
}


void defines(std::ostream& os, const system_information_t& system_description)
{
	// common
	os << "-D_REENTRANT -DPT_NATIVE_WORD=" << WIDTH << " ";

	// os
	if(contains(system_description.os_, SUNOS))
	{
		os << "-DPT_SOLARIS ";
	}
	else if(contains(system_description.os_, LINUX))
	{
		os << "-DPT_LINUX ";
	}
	else if(contains(system_description.os_, MINGW32))
	{
		os << "-DPT_WINDOWS ";
	}

	// arch
	if(contains(system_description.machine_, "sparc"))
	{
		os << "-DPT_SPARC ";		
		if(contains(system_description.machine_details_, "sparcv9")) // SPARC V9
		{
			os << "-DPT_V9 ";
		}
		else if(contains(system_description.machine_details_, "sparcv8plus"))
		{
			os << "-DPT_V8PLUS ";
		}
		else
		{
			os << "-DPT_V8 ";
		}
	}
	else if(contains(system_description.machine_, "i86pc")) // PC compatible
	{
		os << "-DPT_X86 ";
	}
	else if(contains(system_description.machine_, "ia64")) // IA64
	{
		os << "-DPT_IA64 ";
	}
}

void osSpecificDefines(std::ostream& os, const system_information_t& system_description)
{
#ifdef PT_HAVE_LINUX // set in configure.in
	const string version(gnu_get_libc_version());
	if(version.size() >= 3 && version[1] == '.') // make sure it's something like [digit].[digit]*
	{
		if(	version[0] > 2 || 
			(version[0] == 2 && version[2] > 1))
		{
			os << "-DPT_FAST_MALLOC ";
		}
	}
#endif
}

void cxxflags(std::ostream& os, const system_information_t& system_description)
{
	// header directory
	os << "-I" << include << " ";

	// Code generation flag for compiler
	if(contains(system_description.machine_, "ia64"))
	{
		// no flag for IA-64
	}
	else
	{
		os << "-m" << WIDTH << " ";
	}

	// If we are on a SPARC machine in 32-bit mode and the machine supports V8+ 
	// enable V8+ instructions
	if(	WIDTH == 32 &&
		contains(system_description.machine_details_, "sparcv8plus"))
	{
		os << "-mcpu=v9 ";
	}

	// tell compiler to use pthread lib
	if(contains(system_description.os_, SUNOS))
	{
		os  << "-pthreads ";		
	}
	else if(contains(system_description.os_, LINUX))
	{
		os << "-pthread ";
	}
}

void setMode(const OptionValues& values)
{
	// by default 32 bit mode is used
	const string mode = option_as<string>(values, "mode");
	if(mode == "64")
	{
		WIDTH = 64;
	}
	else if(mode == "detect")
	{
		// detect a 64 bit machine		
		if(	contains(system_description.machine_details_, "sparcv9") || // UltraSPARC 
			contains(system_description.machine_details_, "amd64")) // AMD64 or Intel64
		{
			WIDTH = 64;
		}
	}
	if(contains(system_description.machine_, "ia64"))  // Itanium only works in 64bit mode with GCC
	{
		WIDTH = 64;
	}
}



void ldflags(std::ostream& os, const system_information_t& system_description)
{
	return cxxflags(os, system_description);
}


void libs(bool own, bool shared, std::ostream& os, const system_information_t& system_description)
{
	if(own)
	{
		// when using shared lib print -L<libdir> -lportablethreads,
		// otherwise print <libdir>/libportablethreads.a
		if(shared)
		{
			if(contains(system_description.os_, MINGW32))
			{
				os << lib << '/' << "libportablethreads.dll.a ";
			}
			else
			{
				os << "-L" << lib << " -lportablethreads ";
			}
		}
		else
		{
			os << lib << '/' << "libportablethreads.a ";
		}
	}

	if(contains(system_description.os_, SUNOS))
	{
		os << "-lpthread -lrt ";
	}
	else if(contains(system_description.os_, LINUX))
	{
		os << "-lpthread -lrt ";
	}
	else if(contains(system_description.os_, MINGW32))
	{
		os << "-lwinmm ";
	}
}



void setPaths(const OptionValues& values)
{
	if(values.count("lib-dir"))
	{	
		lib = option_as<string>(values, "lib-dir");
		if(!directoryExists(lib))
			throw std::runtime_error("value for option --lib-dir must be a directory!");
	}
	if(values.count("include-dir"))
	{	
		include = option_as<string>(values, "include-dir");
		if(!directoryExists(include))
			throw std::runtime_error("value for option --include-dir must be a directory!");
	}	
}

bool contains(const std::string& haystack, const char* needle)
{
	return haystack.find(needle) != std::string::npos;
}

bool contains(const char* haystack, const char* needle)
{
	return strstr(haystack, needle) != NULL;
}



